﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Threading.Tasks;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;

#if DEBUG
[assembly: System.Runtime.CompilerServices.InternalsVisibleTo("DotNetLibTest")]
#endif

namespace Twitch.EnhancedExperiences
{
    internal partial class Connection : IDisposable
    {
        internal ConnectionException FailingException { get; private set; }
#if DEBUG
        internal static TimeSpan sendDelay = TimeSpan.FromSeconds(1);
        internal static Uri url = new Uri("wss://metadata.twitch.tv/api/ingest");
        internal static Action<string> onDebug;
#else
        private static readonly TimeSpan sendDelay = TimeSpan.FromSeconds(1);
        private static readonly Uri url = new Uri("wss://metadata.twitch.tv/api/ingest");
#endif
        private readonly Accessor accessor;
        private readonly Func<JObject, Task<WebSocket>> connectAsyncFn;
        private readonly Task task;
        private readonly Cancelation cancelation = new Cancelation();
        private volatile WebSocket webSocket;

        public void Dispose() => cancelation.Dispose();

        private Connection(Configuration connectionConfiguration, Func<JObject, Task<WebSocket>> connectAsyncFn, WebSocket webSocket_)
        {
            // Start the communication task.
            accessor = new Accessor(connectionConfiguration.IsConnectTooLarge, connectionConfiguration.InitialData);
            this.connectAsyncFn = connectAsyncFn;
            webSocket = webSocket_;
            task = Task.Run(Communicate);
        }

        internal static async Task<Connection> CreateAsync(Configuration connectionConfiguration)
        {
            var webSocket = await ConnectAsync(connectionConfiguration, connectionConfiguration.InitialData);
            try
            {
                return new Connection(connectionConfiguration, (data) => ConnectAsync(connectionConfiguration, data), webSocket);
            }
            catch
            {
                await SafeCloseAsync(webSocket);
                throw;
            }
        }

        internal async Task DestroyAsync()
        {
            cancelation.Cancel();
            try
            {
                await task;
                if (FailingException == null)
                {
                    await SendRemainingMessages(accessor, webSocket);
                    await Flush(webSocket);
                }
            }
            finally
            {
                webSocket = await SafeCloseAsync(webSocket);
            }
        }

        internal void Access(string path, Accessor.AccessDataFn fn)
        {
            if (FailingException != null)
            {
                throw new ConnectionException(false, FailingException);
            }
            accessor.Access(path, fn);
        }

        private async Task Communicate()
        {
            try
            {
                for (; ; )
                {
                    string message = null;
                    try
                    {
                        // Receive a message if available.
                        message = await webSocket.ReceiveAsync(sendDelay, cancelation);
                        if (message == null)
                        {
                            // The receive operation timed out.  Send any messages.
                            await SendMessages(accessor, webSocket);
                            continue;
                        }

                        // All messages received in this loop are fatal.
                        InvokeOnDebug(message);
#if DEBUG
                        try
                        {
                            var json = JObject.Parse(message);
                            if (json["debug"] != null)
                            {
                                // Ignore debug messages.
                                continue;
                            }
                        }
                        catch (Exception)
                        {
                            // Ignore debug exceptions.
                        }
#endif
                        if (!message.Contains("{\"reconnect\":"))
                        {
                            Debug.WriteLine($"[Connection.Communicate] received unexpected message:  \"{message}\"");
                        }
                    }
                    catch (OperationCanceledException)
                    {
                        break;
                    }
                    catch (Exception ex)
                    {
                        // Most likely a transient networking error has occurred.
                        Debug.WriteLine("[Connection.Communicate] exception:");
                        Debug.WriteLine(ex);
                        InvokeOnDebug(ex);
                    }

                    // Attempt to reconnect to E2.
                    try
                    {
                        await Restart(message);
                    }
                    catch (OperationCanceledException)
                    {
                        break;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.WriteLine("[Connection.Communicate] fatal exception:");
                Debug.WriteLine(ex);
                FailingException = new ConnectionException(false, ex);
            }
        }

        private async Task Restart(string message)
        {
            if (message != null)
            {
                await Flush(webSocket);
            }

            // Close the connection.
            webSocket = await SafeCloseAsync(webSocket);

            if (message != null)
            {
                try
                {
                    var json = JObject.Parse(message);
                    if (json.ContainsKey("reconnect"))
                    {
                        // Extract the reconnection delay from the message.
                        var reconnectDelay = json["reconnect"].Value<int>();

                        // Wait the requested time.
                        await Task.Delay(reconnectDelay, cancelation.Value);
                    }
                }
                catch (OperationCanceledException)
                {
                    throw;
                }
                catch (Exception)
                {
                    // Ignore JSON parsing errors.
                }
            }

            // Re-create the WebSocket connection.
            webSocket = await connectAsyncFn(accessor.ClearMessages());
        }

        private static async Task<WebSocket> ConnectAsync(Configuration connectionConfiguration, JObject data)
        {
            // Create a time-out cancelation.
            using (var timeoutCancelation = connectionConfiguration.Timeout.HasValue ?
                new Cancelation(connectionConfiguration.Timeout.Value) :
                new Cancelation())
            {
                var delay = TimeSpan.FromSeconds(3);
                for (; ; )
                {
                    // Create the WebSocket.
                    var webSocket = new WebSocket();

                    try
                    {
                        // Connect the WebSocket.
                        await webSocket.ConnectAsync(url, timeoutCancelation);

                        for (; ; )
                        {
                            // Send a "connect" message.
                            var connectMessage = connectionConfiguration.CreateConnectMessage(data);
                            await webSocket.SendAsync(connectMessage, timeoutCancelation);

                            // Await a response.
                            var response = await webSocket.ReceiveAsync(timeoutCancelation);
                            InvokeOnDebug(response);
                            var json = JObject.Parse(response);

                            // If the response is "connected", return the WebSocket.
                            if (json.ContainsKey("connected") && json["connected"].Value<bool>())
                            {
                                return webSocket;
                            }

                            // Check for and handle error messages.
                            if (json.ContainsKey("error"))
                            {
                                var error = json["error"];
                                var code = error["code"].Value<string>();

                                // If the response error is "invalid_connect_token", get a new token from
                                // the client and try sending another "connect" message.  If the client
                                // does not provide a new token, cancel this operation.
                                if (code == "invalid_connect_token")
                                {
                                    if (await connectionConfiguration.ProcureToken())
                                    {
                                        continue;
                                    }
                                    throw new ConnectionException(false);
                                }

                                // If the response error is fatal, throw a suitable exception.
                                if (code == "connect_client_not_whitelisted")
                                {
                                    throw new InvalidOperationException(code);
                                }
                                if (code == "connect_invalid_values" || code == "connect_missing_info")
                                {
                                    if (error["error_field"] != null)
                                    {
                                        code += ":  " + error["error_field"].Value<string>();
                                    }
                                    throw new ArgumentException(code, "configuration");
                                }
                            }

                            // For any other response, log it, close the WebSocket, wait a short time,
                            // and try a new connection.
                            Debug.WriteLine("[Connection.ConnectAsync] unexpected response from server:  \"{0}\"", response);
                            await SafeCloseAsync(webSocket);
                            await Task.Delay(delay, timeoutCancelation.Value);
                            delay = TimeSpan.FromSeconds(2 * delay.TotalSeconds + 1);
                            break;
                        }
                    }
                    catch (ConnectionException)
                    {
                        await SafeCloseAsync(webSocket);
                        throw new OperationCanceledException();
                    }
                    catch (OperationCanceledException ex)
                    {
                        await SafeCloseAsync(webSocket);
                        throw new ConnectionException(false, ex);
                    }
                    catch
                    {
                        await SafeCloseAsync(webSocket);
                        throw;
                    }
                }
            }
        }

        private static async Task<WebSocket> SafeCloseAsync(WebSocket webSocket) => webSocket != null ? await webSocket.CloseAsync() : null;

        private static async Task SendMessages(Accessor accessor, WebSocket webSocket)
        {
            // Take operations until reaching the maximum payload size.
            var operations = new List<DeltaOperation>();
            accessor.Access((list) =>
            {
                int n = 0;
                for (; n < list.Count; ++n)
                {
                    operations.Add(list[n]);
                    if (DeltaOperation.IsDeltaTooLarge(operations))
                    {
                        operations.RemoveAt(operations.Count - 1);
                        break;
                    }
                }
                list.RemoveRange(0, n);
            });

            // Send a "delta" message to E2.
            if (operations.Any())
            {
                var deltaMessage = DeltaOperation.MakeDeltaMessage(operations);
                await webSocket.SendAsync(deltaMessage);
            }
        }

        private static async Task SendRemainingMessages(Accessor accessor, WebSocket webSocket)
        {
            try
            {
                var remainingOperations = new List<DeltaOperation>();
                accessor.Access((list) =>
                {
                    remainingOperations.AddRange(list);
                    list.Clear();
                });
                if (remainingOperations.Any())
                {
                    await Task.Delay(sendDelay);
                    var finalMessage = DeltaOperation.MakeDeltaMessage(remainingOperations);
                    if (DeltaOperation.IsDeltaTooLarge(finalMessage))
                    {
                        finalMessage = JObject.FromObject(new { refresh = new { data = accessor.ClearMessages() } });
                    }
                    await webSocket.SendAsync(finalMessage);
                }
            }
            catch (Exception ex)
            {
                Debug.WriteLine("[Connection.SendRemainingMessages] exception:");
                Debug.WriteLine(ex);
            }
        }

        private static async Task Flush(WebSocket webSocket)
        {
            try
            {
                // Send a "flush" message.
                var flushId = Guid.NewGuid().ToString();
                var flushMessage = JObject.FromObject(new { flush = flushId });
                await webSocket.SendAsync(flushMessage);

                // Receive the "flushed" message.  Use a cancelation instead of a time-out
                // so it throws an exception instead of returning null.
                using (var cancelation = new Cancelation(TimeSpan.FromSeconds(9)))
                {
                    await webSocket.ReceiveAsync(cancelation);
                }
            }
            catch
            {
                // Ignore exceptions.
                Debug.WriteLine("[Connection.Flush] warning:  an exception occurred during \"flush\" message processing; data might be lost");
            }
        }

        [Conditional("DEBUG")]
        internal static void InvokeOnDebug(string message)
        {
#if DEBUG
            try
            {
                onDebug?.Invoke(message);
            }
            catch (Exception ex)
            {
                Debug.WriteLine(ex);
            }
#endif
        }

        [Conditional("DEBUG")]
        internal static void InvokeOnDebug(Exception ex)
        {
            while (ex.InnerException != null)
            {
                ex = ex.InnerException;
            }
            InvokeOnDebug("exception:  " + ex.Message);
        }
    }

    internal static class ConnectionExtensions
    {

        internal static Task SendAsync(this WebSocket webSocket, JObject message, Cancelation cancelation = default(Cancelation)) => webSocket.SendAsync(message.ToString(Formatting.None), cancelation);
    }
}
