﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Threading.Tasks;
using Newtonsoft.Json.Linq;

namespace Twitch.EnhancedExperiences
{
    public partial class DataSource
    {
#if DEBUG
        internal Connection connection;
#else
        private Connection connection;
#endif

        /// <summary>
        /// Connect to E2.
        /// </summary>
        /// <param name="configuration">A populated <see cref="Configuration"/>
        /// instance.  See the documentation for required fields.</param>
        /// <returns>Session ID</returns>
        /// <remarks>This method transitions the instance to the "connected" state.</remarks>
        /// <exception cref="ArgumentException">Thrown if one or more of the
        /// configuration values is invalid, such as a null token.</exception>
        /// <exception cref="ConnectionException">Thrown if the instance is
        /// already connected.</exception>
        /// <exception cref="InvalidOperationException">Thrown if the provided
        /// client ID is not allowed to submit data for the provided game ID.</exception>
        /// <exception cref="OperationCanceledException">Thrown if the client
        /// does not provide a token.</exception>
        public Task<string> ConnectAsync(Configuration configuration)
        {
            // Validate the current object state.
            if (connection != null)
            {
                throw new ConnectionException(isConnected: true);
            }

            // Create the connection configuration.
            var connectionConfiguration = new Connection.Configuration(configuration);

            return InternalConnectAsync(connectionConfiguration);
        }

        private async Task<string> InternalConnectAsync(Connection.Configuration connectionConfiguration)
        {
            connection = await Connection.CreateAsync(connectionConfiguration);
            return connectionConfiguration.SessionId;
        }

        /// <summary>
        /// Disconnect from E2.
        /// </summary>
        /// <remarks>This method transitions the instance to the "not connected"
        /// state.</remarks>
        public Task DisconnectAsync()
        {
            ValidateConnection();
            return InternalDisconnectAsync();
        }

        private async Task InternalDisconnectAsync()
        {
            await connection.DestroyAsync();
            connection.Dispose();
            connection = null;
        }

        public void AppendToArrayField<T>(string path, IEnumerable<T> values)
        {
            // Validate the connection.
            ValidateConnection();

            // Create JSON objects from the values.
            var jsons = values.Select(v => new JValue(v)).ToArray();
            if (!jsons.Any())
            {
                Connection.InvokeOnDebug("array is empty");
                Debug.WriteLine("[DataSource.AppendToArrayField] warning:  array is empty; ignoring \"{0}\"", path);
                return;
            }

            // Attempt to access the referenced array.
            connection.Access(path, (data) =>
            {
                var value = data.SelectToken(path);
                if (value != null)
                {
                    if (value is JArray array)
                    {
                        // Append values to the array.
                        foreach (var item in jsons)
                        {
                            array.Add(item);
                        }
                        return new DeltaOperation { Path = path, Type = DeltaOperationType.Append, Value = jsons };
                    }
                    throw new ArgumentException($"\"{path}\" does not specify an array field", nameof(path));
                }
                throw new ArgumentException($"\"{path}\" does not specify a known field", nameof(path));
            });
        }

        public void RemoveField(string path, AbsenceBehavior absenceBehavior = AbsenceBehavior.Error)
        {
            // Validate the connection.
            ValidateConnection();

            // Attempt to remove the field.
            connection.Access(path, (data) =>
            {
                // Validate the path.
                if (path.Last() == ']')
                {
                    throw new ArgumentException($"\"{path}\" does not specify a field", nameof(path));
                }

                // If there is such a field, remove it.
                var value = data.SelectToken(path);
                if (value != null)
                {
                    value.Parent.Remove();
                    return new DeltaOperation { Path = path, Type = DeltaOperationType.Remove };
                }

                // There is no such field.  Use the provided absence behavior to react.
                var message = $"\"{path}\" does not specify a known field";
                if (absenceBehavior == AbsenceBehavior.Error)
                {
                    throw new ArgumentException(message, nameof(path));
                }
                else if (absenceBehavior == AbsenceBehavior.Warning)
                {
                    Debug.WriteLine("[DataSource.RemoveField] warning:  {0}", message);
                }
                return null;
            });
        }

        public void UpdateField(string path, object value)
        {
            // Validate the connection.
            ValidateConnection();

            // If updating the _metadata field, ensure it is valid.
            if (path == "_metadata")
            {
                ValidateData(new { _metadata = value }, nameof(value));
            }

            // Create a JSON representation of the value.
            var json = JToken.FromObject(value);

            // Update the current data.
            connection.Access(path, (data) =>
            {
                // Get a reference to the desired field or array element.
                var currentValue = data.SelectToken(path);
                var isFieldReference = path.Last() != ']';
                if (currentValue == null && isFieldReference)
                {
                    // Get a reference to the parent of the desired field.
                    var parts = path.Split('.');
                    var parent = parts.Length == 1 ? data : data.SelectToken(String.Join(".", parts.Take(parts.Length - 1)));
                    if (parent != null)
                    {
                        if (parent.Type != JTokenType.Object)
                        {
                            throw new ArgumentException($"\"{path}\" does not specify an object field", nameof(path));
                        }

                        // Add a new field to the parent.
                        parent[parts.Last()] = null;
                        currentValue = data.SelectToken(path);
                    }
                }
                if (currentValue == null)
                {
                    var typeName = isFieldReference ? "field" : "array element";
                    throw new ArgumentException($"\"{path}\" does not specify a known {typeName}", nameof(path));
                }

                // Update only if the value changed.
                if (!JToken.DeepEquals(currentValue, json))
                {
                    currentValue.Replace(json);

                    // Enqueue the "Update" operation.
                    return new DeltaOperation { Path = path, Type = DeltaOperationType.Update, Value = json };
                }
                return null;
            });
        }
    }
}
