﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Net.WebSockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace Twitch.EnhancedExperiences
{
    internal class WebSocket
    {
        private readonly ClientWebSocket webSocket = new ClientWebSocket();
        private Task<string> receiveTask;

        internal async Task<WebSocket> CloseAsync()
        {
            try
            {
                // Close the connection.  The Abort invocation ensures the WebSocket is
                // closed.  It has no effect if the CloseOutputAsync invocation successfully
                // closed the WebSocket.
                using (var timeoutCancelation = new Cancelation(TimeSpan.FromSeconds(3)))
                {
                    await webSocket.CloseOutputAsync(WebSocketCloseStatus.NormalClosure, "", timeoutCancelation.Value);
                }
                webSocket.Abort();

                // Drain the receive task.
                try
                {
                    await receiveTask;
                }
                catch (Exception)
                {
                    // Ignore the expected exception.
                }
            }
            catch (Exception ex)
            {
                Debug.WriteLine("[WebSocket.CloseAsync] exception:");
                Debug.WriteLine(ex);
            }
            return null;
        }

        internal async Task ConnectAsync(Uri uri, Cancelation cancelation)
        {
            await webSocket.ConnectAsync(uri, cancelation.Value);
            receiveTask = ReceiveAsync();
        }

        /// <summary>
        /// Receives data as an asynchronous operation.
        /// </summary>
        /// <param name="cancelation">A cancelation used to end this operation early.</param>
        /// <returns>The task object representing the asynchronous operation.</returns>
        /// <remarks>
        /// There are three possible outcomes.
        /// 1. Return a string for a received message.
        /// 2. Throw an OperationCanceledException for a cancelation.
        /// 3. Throw some other exception for an unrecoverable error.
        /// The cancelation affects only this receive operation.  This
        /// differs from the same method in the <see cref="ClientWebSocket"/> class,
        /// which affects the state of the instance of that class.
        /// </remarks>
        internal async Task<string> ReceiveAsync(Cancelation cancelation)
        {
            var source = new TaskCompletionSource<string>();
            using (var registration = cancelation.Register(() => source.TrySetCanceled()))
            {
                var sourceTask = source.Task;
                var task = await Task.WhenAny(receiveTask, sourceTask);
                if (task == receiveTask)
                {
                    // Create a new one to prevent a cancelation race condition.
                    var source_ = new TaskCompletionSource<string>();
                    source_.SetResult(await receiveTask);
                    receiveTask = ReceiveAsync();
                    sourceTask = source_.Task;
                }
                return await sourceTask;
            }
        }

        /// <summary>
        /// Receives data as an asynchronous operation.
        /// </summary>
        /// <param name="timeout">A time-out value used to end this operation early.</param>
        /// <param name="cancelation">A cancelation used to end this operation early.</param>
        /// <returns>The task object representing the asynchronous operation.</returns>
        /// <remarks>
        /// There are four possible outcomes.
        /// 1. Return a string for a received message.
        /// 2. Return null for a time-out.
        /// 3. Throw an OperationCanceledException for a cancelation.
        /// 4. Throw some other exception for an unrecoverable error.
        /// The cancelation affects only this receive operation.  This
        /// differs from the same method in the <see cref="ClientWebSocket"/> class,
        /// which affects the state of the instance of that class.
        /// </remarks>
        internal async Task<string> ReceiveAsync(TimeSpan timeout, Cancelation cancelation = default(Cancelation))
        {
            var timeoutTask = Task.Delay(timeout, cancelation?.Value ?? CancellationToken.None);
            var task = await Task.WhenAny(receiveTask, timeoutTask);
            if (task == receiveTask)
            {
                var message = await receiveTask;
                receiveTask = ReceiveAsync();
                return message;
            }
            await timeoutTask;
            return null;
        }

        /// <summary>
        /// Sends data as an asynchronous operation.
        /// </summary>
        /// <param name="message">The string containing the message to be sent.</param>
        /// <param name="cancelation">A cancelation used to end this operation early.</param>
        /// <returns>The task object representing the asynchronous operation.</returns>
        /// <remarks>
        /// Similar to the same method in the <see cref="ClientWebSocket"/> class,
        /// the cancelation affects the state of the instance of this class.
        /// </remarks>
        internal async Task SendAsync(string message, Cancelation cancelation = default(Cancelation))
        {
            var segment = new ArraySegment<byte>(Encoding.UTF8.GetBytes(message));
            await webSocket.SendAsync(segment, WebSocketMessageType.Text, true, cancelation?.Value ?? CancellationToken.None);
        }

        private async Task<string> ReceiveAsync()
        {
            var receiveBuffer = new ArraySegment<byte>(new byte[2048]);
            for (; ; )
            {
                var response = new List<byte>();
                for (; ; )
                {
                    var result = await webSocket.ReceiveAsync(receiveBuffer, CancellationToken.None);
                    response.AddRange(receiveBuffer.Array.Take(result.Count));
                    if (result.EndOfMessage)
                    {
                        break;
                    }
                }
                var s = Encoding.UTF8.GetString(response.ToArray());
                if (!String.IsNullOrWhiteSpace(s))
                {
                    return s;
                }
            }
        }
    }
}
