﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using Twitch.EnhancedExperiences;

#pragma warning disable 1998 // This enables default debug exception settings.

namespace DotNetLibTest
{
    [TestClass]
    public class AppendTest
    {
        [TestMethod]
        public async Task Fails_ConnectMessageIsTooLarge()
        {
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new { a = new[] { new string('a', 88888) } });
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.AppendToArrayField("a", new[] { new string('b', 11111) }));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_DataSourceIsNotConnected()
        {
            var dataSource = new DataSource();
            await Assert.ThrowsExceptionAsync<ConnectionException>(async () => dataSource.AppendToArrayField("a", new[] { 1 }));
        }

        [TestMethod]
        public async Task Fails_DeltaMessageIsTooLarge()
        {
            var dataSource = await DataSourceTest.CreateAndConnectAsync();
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.AppendToArrayField("a", new[] { new string('a', 22222) }));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_PathIsEmpty()
        {
            var dataSource = await DataSourceTest.CreateAndConnectAsync();
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.AppendToArrayField("", new[] { 1 }));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_PathIsMalformed()
        {
            var dataSource = await DataSourceTest.CreateAndConnectAsync();
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.AppendToArrayField("a[0", new[] { 1 }));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_TargetFieldIsMissing()
        {
            var dataSource = await DataSourceTest.CreateAndConnectAsync();
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.AppendToArrayField("a", new[] { 1 }));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_TargetFieldIsNotAnArray()
        {
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new { a = 1 });
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.AppendToArrayField("a", new[] { 1 }));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Succeeds_TargetArrayIsChanged()
        {
#if DEBUG
            var path = "a";
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new Dictionary<string, int[]> { { path, new[] { 1 } } });
            var source = new TaskCompletionSource<string>();
            Connection.onDebug = source.SetResult;
            var values = new[] { 2 };
            dataSource.AppendToArrayField(path, values);
            var actual = await source.Task;
            await dataSource.DisconnectAsync();
            var expected = "{\"debug\":{\"delta\":[[\"" + path + "\",\"a\"," + new JArray(values).ToString(Formatting.None) + "]]}}";
            Assert.AreEqual(expected, actual);
#endif
        }

        [TestMethod]
        public async Task Succeeds_TargetArrayIsChangedTwice()
        {
#if DEBUG
            var path = "a";
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new Dictionary<string, int[]> { { path, new[] { 1 } } });
            var source = new TaskCompletionSource<string>();
            Connection.onDebug = source.SetResult;
            var values1 = new[] { 2 };
            dataSource.AppendToArrayField(path, values1);
            var values2 = new[] { 3 };
            dataSource.AppendToArrayField(path, values2);
            var actual = await source.Task;
            await dataSource.DisconnectAsync();
            var values = String.Join(",", values1.Concat(values2));
            var expected = "{\"debug\":{\"delta\":[[\"" + path + "\",\"a\",[" + values + "]]]}}";
            Assert.AreEqual(expected, actual);
#endif
        }

        [TestMethod]
        public async Task Succeeds_ValuesArrayIsEmpty()
        {
#if DEBUG
            var path = "a";
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new Dictionary<string, int[]> { { path, new[] { 1 } } });
            var source = new TaskCompletionSource<string>();
            Connection.onDebug = source.SetResult;
            dataSource.AppendToArrayField(path, new int[0]);
            var actual = await source.Task;
            await dataSource.DisconnectAsync();
            var expected = "array is empty";
            Assert.AreEqual(expected, actual);
#endif
        }
    }
}
