﻿using System;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Newtonsoft.Json;
using Twitch.EnhancedExperiences;

namespace DotNetLibTest
{
    [TestClass]
    public class ConnectTest
    {
        [TestMethod]
        public async Task Fails_BroadcastersIsEmptyWithAppToken()
        {
            var dataSource = new DataSource();
            var configuration = DataSourceTest.MakeConfiguration("app");
            configuration.BroadcasterIds = new string[0];
            await Assert.ThrowsExceptionAsync<ArgumentException>(() => dataSource.ConnectAsync(configuration));
        }

        [TestMethod]
        public async Task Fails_BroadcastersIsMissingWithAppToken()
        {
            var dataSource = new DataSource();
            var configuration = DataSourceTest.MakeConfiguration("app");
            await Assert.ThrowsExceptionAsync<ArgumentException>(() => dataSource.ConnectAsync(configuration));
        }

        [TestMethod]
        public async Task Fails_BroadcastersIsNotEmptyWithUserToken()
        {
            var dataSource = new DataSource();
            var configuration = DataSourceTest.MakeConfiguration("user");
            configuration.BroadcasterIds = new[] { "265737932" };
            await Assert.ThrowsExceptionAsync<ArgumentException>(() => dataSource.ConnectAsync(configuration));
        }

        [TestMethod]
        public async Task Fails_CallbackIsMissing()
        {
            var dataSource = new DataSource();
            var configuration = DataSourceTest.MakeConfiguration("bad-token");
            await Assert.ThrowsExceptionAsync<OperationCanceledException>(async () => await dataSource.ConnectAsync(configuration));
        }

        [TestMethod]
        public async Task Fails_ClientIsNotWhitelisted()
        {
            var dataSource = new DataSource();
            var configuration = DataSourceTest.MakeConfiguration("error-connect_client_not_whitelisted");
            await Assert.ThrowsExceptionAsync<InvalidOperationException>(() => dataSource.ConnectAsync(configuration));
        }

        [TestMethod]
        public async Task Fails_ConnectIsInvokedAgain()
        {
            var dataSource = new DataSource();
            var configuration = DataSourceTest.MakeConfiguration("token");
            var sessionId = await dataSource.ConnectAsync(configuration);
            Assert.IsNotNull(sessionId);
            await Assert.ThrowsExceptionAsync<ConnectionException>(async () => await dataSource.ConnectAsync(configuration));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_ConnectMessageIsTooLarge()
        {
            var dataSource = new DataSource();
            var configuration = DataSourceTest.MakeConfiguration("token");
            configuration.InitialData = new { a = new string('a', 111111) };
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => await dataSource.ConnectAsync(configuration));
        }

        [TestMethod]
        public async Task Fails_GoodTokenIsNotProvided()
        {
            var dataSource = new DataSource();
            var configuration = DataSourceTest.MakeConfiguration("bad-token");
            var source = new TaskCompletionSource<bool>();
            configuration.OnTokenExpired = (_) =>
            {
                source.SetResult(true);
                return false;
            };
            await Assert.ThrowsExceptionAsync<OperationCanceledException>(async () => await dataSource.ConnectAsync(configuration));
        }

        [TestMethod]
        public async Task Fails_InitialDataIsNotAnObject()
        {
            var dataSource = new DataSource();
            var configuration = DataSourceTest.MakeConfiguration("token");
            configuration.InitialData = 1;
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => await dataSource.ConnectAsync(configuration));
        }

        [TestMethod]
        public async Task Fails_JsonFromServerIsInvalid()
        {
            var dataSource = new DataSource();
            var token = "unexpected-payload-1";
            var configuration = DataSourceTest.MakeConfiguration(token);
            await Assert.ThrowsExceptionAsync<JsonReaderException>(async () => await dataSource.ConnectAsync(configuration));
        }

        [TestMethod]
        public async Task Fails_MetadataIsNotAnObject()
        {
            var dataSource = new DataSource();
            var configuration = DataSourceTest.MakeConfiguration("token");
            configuration.InitialData = new { _metadata = 1 };
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => await dataSource.ConnectAsync(configuration));
        }

        [TestMethod]
        public async Task Fails_PayloadFromServerIsNotJson()
        {
            var dataSource = new DataSource();
            var token = "unexpected-payload-$";
            var configuration = DataSourceTest.MakeConfiguration(token);
            await Assert.ThrowsExceptionAsync<JsonReaderException>(async () => await dataSource.ConnectAsync(configuration));
        }

        [TestMethod]
        public async Task Fails_TokenIsEmpty()
        {
            var dataSource = new DataSource();
            var configuration = DataSourceTest.MakeConfiguration("");
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => await dataSource.ConnectAsync(configuration));
        }

        [TestMethod]
        public async Task Fails_TokenIsMissing()
        {
            var dataSource = new DataSource();
            var configuration = DataSourceTest.MakeConfiguration();
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => await dataSource.ConnectAsync(configuration));
        }

        [TestMethod]
        public async Task Succeeds_BroadcastersIsEmptyWithUserToken()
        {
            var dataSource = new DataSource();
            var configuration = DataSourceTest.MakeConfiguration("user");
            configuration.BroadcasterIds = new string[0];
            var sessionId = await dataSource.ConnectAsync(configuration);
            Assert.IsNotNull(sessionId);
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Succeeds_BroadcastersIsMissingWithUserToken()
        {
            var dataSource = new DataSource();
            var configuration = DataSourceTest.MakeConfiguration("user");
            var sessionId = await dataSource.ConnectAsync(configuration);
            Assert.IsNotNull(sessionId);
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Succeeds_BroadcastersIsNotEmptyWithAppToken()
        {
            var dataSource = new DataSource();
            var configuration = DataSourceTest.MakeConfiguration("app");
            configuration.BroadcasterIds = new[] { "265737932" };
            var sessionId = await dataSource.ConnectAsync(configuration);
            Assert.IsNotNull(sessionId);
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Succeeds_ConnectionIsEstablishedAfterAbort()
        {
#if DEBUG
            var dataSource = new DataSource();
            var token = "abort";
            var configuration = DataSourceTest.MakeConfiguration(token);
            var sources = Enumerable.Range(0, 3).Select(_ => new TaskCompletionSource<string>()).ToArray();
            Connection.onDebug = (s) =>
            {
                sources.First((source) => !source.Task.IsCompleted).SetResult(s);
            };
            var sessionId = await dataSource.ConnectAsync(configuration);
            Assert.IsNotNull(sessionId);
            var task = Task.WhenAny(sources.Last().Task, Task.Delay(TimeSpan.FromSeconds(3)));
            await task;
            await dataSource.DisconnectAsync();
            Assert.AreEqual(sources.Last().Task, task.Result);
            Assert.AreEqual("{\"connected\":true}", sources[0].Task.Result);
            Assert.AreEqual("exception:  ", sources[1].Task.Result.Substring(0, 12));
            Assert.AreEqual("{\"connected\":true}", sources[2].Task.Result);
#endif
        }

        [TestMethod]
        public async Task Succeeds_ConnectionIsEstablishedAfterLongRestart()
        {
#if DEBUG
            var dataSource = new DataSource();
            var token = "restart-long";
            var configuration = DataSourceTest.MakeConfiguration(token);
            var sources = Enumerable.Range(0, 3).Select(_ => new TaskCompletionSource<string>()).ToArray();
            Connection.onDebug = (s) =>
            {
                sources.First((source) => !source.Task.IsCompleted).SetResult(s);
            };
            var sessionId = await dataSource.ConnectAsync(configuration);
            Assert.IsNotNull(sessionId);
            var task = Task.WhenAny(sources.Last().Task, Task.Delay(TimeSpan.FromSeconds(3)));
            await task;
            await dataSource.DisconnectAsync();
            Assert.AreEqual(sources.Last().Task, task.Result);
            Assert.AreEqual("{\"connected\":true}", sources[0].Task.Result);
            Assert.AreEqual("{\"reconnect\":2200}", sources[1].Task.Result);
            Assert.AreEqual("{\"connected\":true}", sources[2].Task.Result);
#endif
        }

        [TestMethod]
        public async Task Succeeds_ConnectionIsEstablishedAfterShortRestart()
        {
#if DEBUG
            var dataSource = new DataSource();
            var token = "restart-short";
            var configuration = DataSourceTest.MakeConfiguration(token);
            var sources = Enumerable.Range(0, 3).Select(_ => new TaskCompletionSource<string>()).ToArray();
            Connection.onDebug = (s) =>
            {
                sources.First((source) => !source.Task.IsCompleted).SetResult(s);
            };
            var sessionId = await dataSource.ConnectAsync(configuration);
            Assert.IsNotNull(sessionId);
            var task = Task.WhenAny(sources.Last().Task, Task.Delay(TimeSpan.FromSeconds(1)));
            await task;
            await dataSource.DisconnectAsync();
            Assert.AreEqual(sources.Last().Task, task.Result);
            Assert.AreEqual("{\"connected\":true}", sources[0].Task.Result);
            Assert.AreEqual("{\"reconnect\":110}", sources[1].Task.Result);
            Assert.AreEqual("{\"connected\":true}", sources[2].Task.Result);
#endif
        }

        [TestMethod]
        public async Task Succeeds_InvalidJsonFromServerIsThrown()
        {
            var dataSource = new DataSource();
            var token = "unexpected-payload-$";
            var configuration = DataSourceTest.MakeConfiguration(token);
            await Assert.ThrowsExceptionAsync<JsonReaderException>(async () => await dataSource.ConnectAsync(configuration));
        }

        [TestMethod]
        public async Task Succeeds_TokenRequestIsFulfilledAsynchronously()
        {
            bool OnTokenExpired(DataSource.TokenRefreshFn fn)
            {
                Task.Run(async () =>
                {
                    await Task.Delay(TimeSpan.FromSeconds(.22));
                    fn("token");
                });
                return true;
            }
            var dataSource = new DataSource();
            var configuration = DataSourceTest.MakeConfiguration("bad-token");
            configuration.OnTokenExpired = OnTokenExpired;
            var sessionId = await dataSource.ConnectAsync(configuration);
            Assert.IsNotNull(sessionId);
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Succeeds_TokenRequestIsFulfilledSynchronously()
        {
            var dataSource = new DataSource();
            var configuration = DataSourceTest.MakeConfiguration("bad-token");
            bool OnTokenExpired(DataSource.TokenRefreshFn fn)
            {
                fn("token");
                return true;
            }
            configuration.OnTokenExpired = OnTokenExpired;
            var sessionId = await dataSource.ConnectAsync(configuration);
            Assert.IsNotNull(sessionId);
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Succeeds_UnexpectedJsonFromServerIsThrown()
        {
            var dataSource = new DataSource();
            var token = "unexpected-payload-1";
            var configuration = DataSourceTest.MakeConfiguration(token);
            await Assert.ThrowsExceptionAsync<JsonReaderException>(async () => await dataSource.ConnectAsync(configuration));
        }
    }
}
