﻿using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Twitch.EnhancedExperiences;

#pragma warning disable 1998 // This enables default debug exception settings.

namespace DotNetLibTest
{
    [TestClass]
    public class RemoveTest
    {
        [TestMethod]
        public async Task Fails_DataSourceIsNotConnected()
        {
            var dataSource = new DataSource();
            await Assert.ThrowsExceptionAsync<ConnectionException>(async () => dataSource.RemoveField("a"));
        }

        [TestMethod]
        public async Task Fails_FieldIsNotFound()
        {
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new { a = new { b = 1 } });
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.RemoveField("a.c"));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_PathIsEmpty()
        {
            var dataSource = await DataSourceTest.CreateAndConnectAsync();
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.RemoveField(""));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_PathIsMalformed()
        {
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new { a = new[] { 1 } });
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.RemoveField("a[0"));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_RemoveIsInvokedTwice()
        {
            var path = "a";
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new Dictionary<string, int> { { path, 1 } });
            dataSource.RemoveField(path);
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.RemoveField(path));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_TargetIsArrayElement()
        {
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new { a = new[] { 1, 2 } });
            var path = "a[1]";
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.RemoveField(path));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Succeeds_FieldIsRemoved()
        {
#if DEBUG
            var path = "a";
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new Dictionary<string, int> { { path, 1 } });
            var source = new TaskCompletionSource<string>();
            Connection.onDebug = source.SetResult;
            dataSource.RemoveField(path);
            var actual = await source.Task;
            await dataSource.DisconnectAsync();
            var expected = "{\"debug\":{\"delta\":[[\"" + path + "\"]]}}";
            Assert.AreEqual(expected, actual);
#endif
        }

        [TestMethod]
        public async Task Succeeds_FieldIsRemovedAfterAppend()
        {
#if DEBUG
            var path = "a";
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new Dictionary<string, int[]> { { path, new[] { 1 } } });
            var source = new TaskCompletionSource<string>();
            Connection.onDebug = source.SetResult;
            var values = new[] { 2 };
            dataSource.AppendToArrayField(path, values);
            dataSource.RemoveField(path);
            var task = await Task.WhenAny(source.Task, Task.Delay(2 * DataSourceTest.SendDelay));
            Assert.AreEqual(source.Task, task);
            var expected = "{\"debug\":{\"delta\":[[\"" + path + "\"]]}}";
            var actual = await source.Task;
            Assert.AreEqual(expected, actual);
            await dataSource.DisconnectAsync();
#endif
        }

        [TestMethod]
        public async Task Succeeds_FieldIsRemovedAfterUpdate()
        {
#if DEBUG
            var path = "a";
            var value = 1;
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new Dictionary<string, int[]> { { path, new[] { value + 1 } } });
            var source = new TaskCompletionSource<string>();
            Connection.onDebug = source.SetResult;
            dataSource.UpdateField(path, value);
            dataSource.RemoveField(path);
            var task = await Task.WhenAny(source.Task, Task.Delay(2 * DataSourceTest.SendDelay));
            Assert.AreEqual(source.Task, task);
            var expected = "{\"debug\":{\"delta\":[[\"" + path + "\"," + value.ToString() + "],[\"" + path + "\"]]}}";
            var actual = await source.Task;
            Assert.AreEqual(expected, actual);
            await dataSource.DisconnectAsync();
#endif
        }

        [TestMethod]
        public async Task Succeeds_MetadataIsRemoved()
        {
#if DEBUG
            var path = "_metadata";
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new Dictionary<string, object> { { path, new { value = 1 } } });
            var source = new TaskCompletionSource<string>();
            Connection.onDebug = source.SetResult;
            dataSource.RemoveField(path);
            var actual = await source.Task;
            await dataSource.DisconnectAsync();
            var expected = "{\"debug\":{\"delta\":[[\"" + path + "\"]]}}";
            Assert.AreEqual(expected, actual);
#endif
        }

        [TestMethod]
        public async Task Succeeds_OneFieldIsRemovedAfterAnother()
        {
#if DEBUG
            var path1 = "a";
            var path2 = "b";
            var data = new Dictionary<string, int> { { path1, 1 }, { path2, 2 } };
            var dataSource = await DataSourceTest.CreateAndConnectAsync(data);
            var source = new TaskCompletionSource<string>();
            Connection.onDebug = source.SetResult;
            dataSource.RemoveField(path1);
            dataSource.RemoveField(path2);
            var actual = await source.Task;
            await dataSource.DisconnectAsync();
            var expected = "{\"debug\":{\"delta\":[[\"" + path1 + "\"],[\"" + path2 + "\"]]}}";
            Assert.AreEqual(expected, actual);
#endif
        }
    }
}
