﻿using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using Twitch.EnhancedExperiences;

#pragma warning disable 1998 // This enables default debug exception settings.

namespace DotNetLibTest
{
    [TestClass]
    public class UpdateTest
    {
        [TestMethod]
        public async Task Fails_ArrayIndexIsOutOfBounds()
        {
            var name = "a";
            var values = new[] { 1 };
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new Dictionary<string, int[]> { { name, values } });
            var path = $"{name}[{values.Length}]";
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.UpdateField(path, 0));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_ConnectMessageIsTooLarge()
        {
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new { a = new string('a', 88888) });
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.UpdateField("b", new string('b', 11111)));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_DataSourceIsNotConnected()
        {
            var dataSource = new DataSource();
            await Assert.ThrowsExceptionAsync<ConnectionException>(async () => dataSource.UpdateField("a", 1));
        }

        [TestMethod]
        public async Task Fail_DeltasMessageIsTooLarge()
        {
            var dataSource = await DataSourceTest.CreateAndConnectAsync();
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.UpdateField("a", new string('a', 22222)));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_MetadataIsNotAnObject()
        {
            var dataSource = await DataSourceTest.CreateAndConnectAsync();
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.UpdateField("_metadata", 1));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_PathIsEmpty()
        {
            var dataSource = await DataSourceTest.CreateAndConnectAsync();
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.UpdateField("", 0));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_PathIsMalformed()
        {
            var dataSource = await DataSourceTest.CreateAndConnectAsync();
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.UpdateField("a[0", 1));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_TargetFieldIsAnArray()
        {
            var path = "a";
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new Dictionary<string, int[]> { { path, new[] { 1 } } });
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.UpdateField($"{path}.b", 0));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_TargetFieldIsMissing()
        {
            var dataSource = await DataSourceTest.CreateAndConnectAsync();
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.UpdateField("a.b", 1));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Fails_TargetFieldIsNotAnArray()
        {
            var path = "a";
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new Dictionary<string, int> { { path, 1 } });
            await Assert.ThrowsExceptionAsync<ArgumentException>(async () => dataSource.UpdateField($"{path}[0]", 0));
            await dataSource.DisconnectAsync();
        }

        [TestMethod]
        public async Task Succeeds_ArrayElementIsChanged()
        {
#if DEBUG
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new
            {
                foo = new
                {
                    bar = new[] { new[] {
                        new { baz = 1 },
                        new { baz = 2 },
                    } }
                }
            });
            var source = new TaskCompletionSource<string>();
            Connection.onDebug = source.SetResult;
            var path = "foo.bar[0][1]";
            var value = 3;
            dataSource.UpdateField(path, value);
            var actual = await source.Task;
            await dataSource.DisconnectAsync();
            var expected = "{\"debug\":{\"delta\":[[\"" + path + "\"," + value + "]]}}";
            Assert.AreEqual(expected, actual);
#endif
        }

        [TestMethod]
        public async Task Succeeds_FieldIsAdded()
        {
#if DEBUG
            var dataSource = await DataSourceTest.CreateAndConnectAsync();
            var source = new TaskCompletionSource<string>();
            Connection.onDebug = source.SetResult;
            var path = "a";
            var value = 1;
            dataSource.UpdateField(path, value);
            var actual = await source.Task;
            await dataSource.DisconnectAsync();
            var expected = "{\"debug\":{\"delta\":[[\"" + path + "\"," + value + "]]}}";
            Assert.AreEqual(expected, actual);
#endif
        }

        [TestMethod]
        public async Task Succeeds_FieldIsAddedAfterRemove()
        {
#if DEBUG
            var path = "a";
            var value = 1;
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new Dictionary<string, int> { { path, value + 1 } });
            var source = new TaskCompletionSource<string>();
            Connection.onDebug = source.SetResult;
            dataSource.RemoveField(path);
            dataSource.UpdateField(path, value);
            var actual = await source.Task;
            await dataSource.DisconnectAsync();
            var expected = "{\"debug\":{\"delta\":[[\"" + path + "\"," + value + "]]}}";
            Assert.AreEqual(expected, actual);
#endif
        }

        [TestMethod]
        public async Task Succeeds_FieldIsChanged()
        {
#if DEBUG
            var path = "a";
            var value = 1;
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new Dictionary<string, int> { { path, value + 1 } });
            var source = new TaskCompletionSource<string>();
            Connection.onDebug = source.SetResult;
            dataSource.UpdateField(path, value);
            var actual = await source.Task;
            await dataSource.DisconnectAsync();
            var expected = "{\"debug\":{\"delta\":[[\"" + path + "\"," + value + "]]}}";
            Assert.AreEqual(expected, actual);
#endif
        }

        [TestMethod]
        public async Task Succeeds_MetadataIsChanged()
        {
#if DEBUG
            var dataSource = await DataSourceTest.CreateAndConnectAsync();
            var source = new TaskCompletionSource<string>();
            Connection.onDebug = source.SetResult;
            var value = new { a = 1 };
            dataSource.UpdateField("_metadata", value);
            var actual = await source.Task;
            await dataSource.DisconnectAsync();
            var expected = "{\"debug\":{\"delta\":[[\"_metadata\"," + JObject.FromObject(value).ToString(Formatting.None) + "]]}}";
            Assert.AreEqual(expected, actual);
#endif
        }

        [TestMethod]
        public async Task Succeeds_OneFieldIsUpdatedAfterAnother()
        {
#if DEBUG
            var dataSource = await DataSourceTest.CreateAndConnectAsync();
            var source = new TaskCompletionSource<string>();
            Connection.onDebug = source.SetResult;
            var path1 = "a";
            var value1 = 1;
            dataSource.UpdateField(path1, value1);
            var path2 = "b";
            var value2 = 2;
            dataSource.UpdateField(path2, value2);
            var actual = await source.Task;
            await dataSource.DisconnectAsync();
            var expected = "{\"debug\":{\"delta\":[[\"" + path1 + "\"," + value1 + "],[\"" + path2 + "\"," + value2 + "]]}}";
            Assert.AreEqual(expected, actual);
#endif
        }

        [TestMethod]
        public async Task Succeeds_UpdateIsInvokedTwice()
        {
#if DEBUG
            var path = "a";
            var value = 1;
            var dataSource = await DataSourceTest.CreateAndConnectAsync(new Dictionary<string, int> { { path, value + 1 } });
            var source = new TaskCompletionSource<string>();
            Connection.onDebug = source.SetResult;
            dataSource.UpdateField(path, value + 2);
            dataSource.UpdateField(path, value);
            var actual = await source.Task;
            await dataSource.DisconnectAsync();
            var expected = "{\"debug\":{\"delta\":[[\"" + path + "\"," + value + "]]}}";
            Assert.AreEqual(expected, actual);
#endif
        }
    }
}
