import json
from collections import deque
from datetime import datetime, timedelta
from enum import Enum, auto
from itertools import dropwhile
from typing import Any, Iterable, Optional

maximum_delta_size = 20000

def is_too_large(deltas: list) -> bool:
    return len(json.dumps({ 'delta': deltas })) > maximum_delta_size

class MessageType(Enum):
    Append = auto()
    Reauthorize = auto()
    Reconnect = auto()
    Refresh = auto()
    Remove = auto()
    Update = auto()

# These are in priority order.
unique_message_types = [MessageType.Reconnect, MessageType.Reauthorize, MessageType.Refresh]

# This is a stand-in for QueueMessage, not yet defined.
QueueMessage = Any

class QueueMessage:
    Refresh = None

    def __init__(self, type: MessageType, path: str=None, value: Any=None):
        self.type = type
        self.path = path
        self.value = value
        if not any(type == t for t in unique_message_types) and is_too_large(self.as_delta()):
            raise RuntimeError(f'delta is too large for "{path or type}"')

    def make_reauthorize(*, clearing_token: bool) -> QueueMessage:
        return QueueMessage(MessageType.Reauthorize, value=clearing_token)

    def make_reconnect(reconnect_delay: timedelta=timedelta()) -> QueueMessage:
        reconnect_time = reconnect_delay + datetime.now()
        return QueueMessage(MessageType.Reconnect, value=reconnect_time)

    def as_delta(self) -> list:
        if self.type == MessageType.Append:
            return [self.path, 'a', self.value]
        if self.type == MessageType.Remove:
            return [self.path]
        if self.type == MessageType.Update:
            return [self.path, self.value]
        raise RuntimeError(f'unexpected message type "{self.type}"')

QueueMessage.Refresh = QueueMessage(MessageType.Refresh)

class MessageQueue:
    def __init__(self):
        self.__queue = deque()

    def __bool__(self) -> bool:
        return bool(self.__queue)

    def clear(self) -> None:
        self.__queue.clear()

    def dequeue(self) -> QueueMessage:
        return self.__queue.popleft()

    def find_unique_message(self) -> Optional[QueueMessage]:
        for unique_message_type in unique_message_types:
            unique_message = next(dropwhile(lambda m: m.type != unique_message_type, self.__queue), None)
            if unique_message:
                return unique_message

    def peek(self) -> QueueMessage:
        return self.__queue[0]

    def enqueue(self, *messages: Iterable[QueueMessage]) -> None:
        self.__queue.extend(messages)

    def replace_with(self, *messages: Iterable[QueueMessage]) -> None:
        self.__queue.clear()
        self.enqueue(*messages)
