import json
from __init__ import DataSource, logger
from threading import Condition

def _configure(send_delay, url):
    import __init__ as e2
    e2.send_delay = send_delay
    e2.url = url

def _create_configuration(*, broadcaster_ids=['1'], initial_data=None, on_token_expired=None, token='app'):
    initial_data = initial_data or {
      'foo': {
        'bar': [[{ 'one': [1, 2] },
          { 'two': [3, 4] },]],
      },
    }
    def on_token_expired_(_fn):
        print('unexpected token refresh call-back')
        return False
    on_token_expired = on_token_expired or on_token_expired_
    return {
        'broadcaster_ids': broadcaster_ids,
        'is_debug': True,
        'environment': 'dev',
        'game_id': 'test',
        'initial_data': initial_data,
        'on_token_expired': on_token_expired,
        'token': token,
    }

def _connect(expected, configuration, wants_connection=False):
    data_source = DataSource()
    cv = Condition()
    actual = []
    def log_fn(*args, exc_info=None):
        actual.extend(args)
        if exc_info:
            actual.append(exc_info)
        with cv:
            cv.notify()
    logger.error = log_fn
    session_id = data_source.connect(**configuration)
    if type(session_id) is str:
        with cv:
            cv.wait_for(lambda: actual)
        if actual != expected:
            print('actual:', len(actual), *actual)
            print('expected:', len(expected), *expected)
            return None
    else:
        print(f'unexpected session_id "{session_id}"')
        return None
    if wants_connection:
        return data_source
    data_source.disconnect()

def _create_and_connect(initial_data, fn, expected):
    actual = []
    cv = Condition()
    def log_fn(*args, exc_info=None):
        actual.extend(args)
        if exc_info:
            actual.append(exc_info)
        with cv:
            cv.notify()
    data_source = DataSource()
    configuration = _create_configuration(initial_data=initial_data)
    logger.error = log_fn
    session_id = data_source.connect(**configuration)
    if type(session_id) is not str:
        print(f'unexpected session_id "{session_id}"')
    with cv:
        cv.wait(timeout=1.1)
    expected_connect = ['[DataSource.on_message] unexpected response from server:  "%s"', 'connected']
    if actual != expected_connect:
        print('actual:', len(actual), *actual)
        print('expected:', len(expected_connect), *expected_connect)
    actual.clear()
    logger.warning = log_fn
    fn(data_source)
    with cv:
        cv.wait(timeout=1.1)
    data_source.disconnect()
    if actual != expected:
        print('actual:', len(actual), *actual)
        print('expected:', len(expected), *expected)

def _create_connect_and_check_assertion(*, initial_data, expected, fn, error_type=AssertionError):
    path = ''
    def log_fn(*args, exc_info=None):
        with cv:
            cv.notify()
    logger.error = log_fn
    configuration = _create_configuration(initial_data=initial_data)
    data_source = DataSource()
    cv = Condition()
    session_id = data_source.connect(**configuration)
    if type(session_id) is str:
        with cv:
            if not cv.wait(timeout=1.1):
                print('timout expired')
            else:
                try:
                    fn(data_source)
                except error_type as ex:
                    actual = list(ex.args)
                    if actual != expected:
                        print('actual:', len(actual), *actual)
                        print('expected:', len(expected), *expected)
                except BaseException as ex:
                    print(f'actual {type(ex)}:', len(ex.args), *ex.args)
                    print(f'expected {error_type}:', len(expected), *expected)
    else:
        print(f'unexpected session_id "{session_id}"')
    data_source.disconnect()

def _fail_connect_due_to_initial_data(initial_data, expected):
    configuration = _create_configuration()
    configuration['initial_data'] = initial_data
    data_source = DataSource()
    try:
        data_source.connect(**configuration)
        print('unexpected success')
        data_source.disconnect()
    except Exception as ex:
        actual = list(ex.args)
        expected = [expected]
        if actual != expected:
            print('actual:', len(actual), *actual)
            print('expected:', len(expected), *expected)

def connect_succeeds():
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', 'connected']
    configuration = _create_configuration()
    _connect(expected, configuration)

def connect_with_user_token_succeeds():
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', 'connected']
    configuration = _create_configuration(broadcaster_ids=None)
    configuration['token'] = 'user'
    _connect(expected, configuration)

def connect_with_empty_broadcaster_ID_array_fails():
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', {'error': {'code': 'connect_invalid_values', 'message': 'connect message has an invalid value for a required field', 'error_field': 'broadcaster_ids'}}]
    configuration = _create_configuration(broadcaster_ids=[])
    _connect(expected, configuration)

def connect_with_no_broadcaster_ID_array_fails():
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', {'error': {'code': 'connect_invalid_values', 'message': 'connect message has an invalid value for a required field', 'error_field': 'broadcaster_ids'}}]
    configuration = _create_configuration(broadcaster_ids=None)
    _connect(expected, configuration)

def connect_with_immediate_token_acquisition_succeeds():
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', 'connected']
    def on_token_expired(fn):
        fn('token')
        return True
    configuration = _create_configuration(broadcaster_ids=None, on_token_expired=on_token_expired, token='')
    _connect(expected, configuration)

def connect_with_delayed_token_acquisition_succeeds():
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', 'connected']
    def on_token_expired(fn):
        def run():
            import time
            time.sleep(.11)
            fn('token')
        from threading import Thread
        Thread(target=run).start()
        return True
    configuration = _create_configuration(broadcaster_ids=None, on_token_expired=on_token_expired, token='')
    _connect(expected, configuration)

def connect_fails_due_to_token():
    def on_token_expired(_fn):
        with cv:
            cv.notify()
        return False
    configuration = _create_configuration(broadcaster_ids=None, on_token_expired=on_token_expired, token='bad-token')
    data_source = DataSource()
    cv = Condition()
    session_id = data_source.connect(**configuration)
    if type(session_id) is str:
        with cv:
            if not cv.wait(timeout=1.1):
                print('timout expired')
    else:
        print(f'unexpected session_id "{session_id}"')

def connect_fails_due_to_current_connection():
    configuration = _create_configuration()
    data_source = DataSource()
    session_id = data_source.connect(**configuration)
    try:
        data_source.connect(**configuration)
        print('unexpected success')
    except Exception as ex:
        actual = list(ex.args)
        expected = ['already connected']
        if actual != expected:
            print('actual:', len(actual), *actual)
            print('expected:', len(expected), *expected)
    data_source.disconnect()

def connect_fails_due_to_not_an_object():
    _fail_connect_due_to_initial_data('$', 'data is not a dictionary')

def connect_fails_due_to_size():
    _fail_connect_due_to_initial_data({ 'large': ['a'] * 99999 }, 'initial data object is too large')

def connect_fails_due_to_invalid_metadata():
    _fail_connect_due_to_initial_data({ '_metadata': 1 }, '_metadata field is not a dictionary')

def component_exposes_correct_members():
    expected = ['append_to_list_field', 'connect', 'disconnect', 'remove_field', 'update_field']
    g = dir(DataSource())
    g = (s for s in g if not (s.startswith('__') and s.endswith('__')))
    g = (s for s in g if not s.startswith('_DataSource__'))
    actual = sorted(g)
    if actual != expected:
        print('actual:', len(actual), *actual)
        print('expected:', len(expected), *expected)

def component_handles_invalid_JSON_from_server():
    def log_fn(*args, exc_info=None):
        actual.extend(args)
        if exc_info:
            actual.append(exc_info)
        with cv:
            cv.notify()
    logger.error = log_fn
    configuration = _create_configuration(token='unexpected-payload')
    configuration['game_id'] = '$'
    data_source = DataSource()
    cv = Condition()
    actual = []
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', '$']
    session_id = data_source.connect(**configuration)
    if type(session_id) is str:
        with cv:
            cv.wait_for(lambda: actual)
        if actual != expected:
            print('actual:', len(actual), *actual)
            print('expected:', len(expected), *expected)
    else:
        print(f'unexpected session_id "{session_id}"')
    data_source.disconnect()

def component_handles_unexpected_JSON_from_server():
    def log_fn(*args, exc_info=None):
        actual.extend(args)
        if exc_info:
            actual.append(exc_info)
        with cv:
            cv.notify()
    logger.error = log_fn
    configuration = _create_configuration(token='unexpected-payload')
    configuration['game_id'] = '1'
    data_source = DataSource()
    cv = Condition()
    actual = []
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', 1]
    session_id = data_source.connect(**configuration)
    if type(session_id) is str:
        with cv:
            cv.wait_for(lambda: actual)
        if actual != expected:
            print('actual:', len(actual), *actual)
            print('expected:', len(expected), *expected)
    else:
        print(f'unexpected session_id "{session_id}"')
    data_source.disconnect()

def component_aborts_successfully():
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', 'connected']
    configuration = _create_configuration()
    configuration['token'] = 'abort'
    _connect(expected, configuration)

def component_reconnects_after_long_wait():
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', 'connected']
    configuration = _create_configuration()
    configuration['game_id'] = '2200'
    _connect(expected, configuration)


def component_reconnects_after_short_wait():
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', 'connected']
    configuration = _create_configuration()
    configuration['game_id'] = '110'
    _connect(expected, configuration)

def component_handles_unexpected_connect():
    actual = []
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', {'connected': False}]
    def log_fn(*args, exc_info=None):
        actual.extend(args)
        if exc_info:
            actual.append(exc_info)
        with cv:
            cv.notify()
    logger.error = log_fn
    configuration = _create_configuration(token='unexpected-payload')
    configuration['game_id'] = '{"connected":false}'
    data_source = DataSource()
    cv = Condition()
    session_id = data_source.connect(**configuration)
    if type(session_id) is str:
        with cv:
            if not cv.wait(timeout=1.1):
                print('timout expired')
            elif actual != expected:
                print('actual:', len(actual), *actual)
                print('expected:', len(expected), *expected)
    else:
        print(f'unexpected session_id "{session_id}"')
    data_source.disconnect()

def component_handles_unexpected_reconnect():
    actual = []
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', {'reconnect': None}]
    def log_fn(*args, exc_info=None):
        actual.extend(args)
        if exc_info:
            actual.append(exc_info)
        with cv:
            cv.notify()
    logger.error = log_fn
    configuration = _create_configuration(token='unexpected-payload')
    configuration['game_id'] = '{"reconnect":null}'
    data_source = DataSource()
    cv = Condition()
    session_id = data_source.connect(**configuration)
    if type(session_id) is str:
        with cv:
            if not cv.wait(timeout=1.1):
                print('timout expired')
            elif actual != expected:
                print('actual:', len(actual), *actual)
                print('expected:', len(expected), *expected)
    else:
        print(f'unexpected session_id "{session_id}"')
    data_source.disconnect()

def append_succeeds():
    path, values = 'a', [2]
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', { 'debug': { 'delta': [[path, 'a', values]] }}]
    def fn(data_source):
        data_source.append_to_list_field(path, values)
    _create_and_connect({ path: [1] }, fn, expected)

def append_empty_succeeds():
    path, values = 'a', []
    expected = ['[DataSource.append_to_list_field] values is empty; ignoring path "%s"', path]
    def fn(data_source):
        data_source.append_to_list_field(path, values)
    _create_and_connect({ path: [1] }, fn, expected)

def double_append_succeeds():
    path, values1, values2 = 'a', [2], [3]
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', { 'debug': { 'delta': [[path, 'a', values1 + values2]] }}]
    def fn(data_source):
        data_source.append_to_list_field(path, values1)
        data_source.append_to_list_field(path, values2)
    _create_and_connect({ path: [1] }, fn, expected)

def append_then_remove_succeeds():
    path, values = 'a', [2]
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', { 'debug': { 'delta': [[path, 'a', values]] }}]
    def fn(data_source):
        data_source.append_to_list_field(path, values)
    _create_and_connect({ path: [1] }, fn, expected)

def append_fails_due_to_no_connection():
    expected = ['connection not established']
    configuration = _create_configuration(initial_data={ 'a': [1] })
    data_source = DataSource()
    try:
        data_source.append_to_list_field('a', [1])
        print('unexpected success')
    except AssertionError as ex:
        actual = list(ex.args)
        if actual != expected:
            print('actual:', len(actual), *actual)
            print('expected:', len(expected), *expected)
    except BaseException as ex:
        print(f'actual {type(ex)}:', len(ex.args), *ex.args)
        print('expected AssertionError:', len(expected), *expected)

def append_fails_due_to_empty_path():
    def fn(data_source):
        data_source.append_to_list_field('', [2])
    _create_connect_and_check_assertion(initial_data={ 'a': [1] }, expected=['path is empty'], fn=fn)

def append_fails_due_to_invalid_path_type():
    def fn(data_source):
        data_source.append_to_list_field(1, [2])
    _create_connect_and_check_assertion(initial_data={ 'a': [1] }, expected=['path is not a string'], fn=fn)

def append_fails_due_to_invalid_path():
    path = 'a[0'
    def fn(data_source):
        data_source.append_to_list_field(path, [2])
    expected = [f'"{path}" is not a valid field specifier']
    _create_connect_and_check_assertion(initial_data={ 'a': [1] }, expected=expected, fn=fn)

def append_fails_due_to_no_field():
    path = 'b'
    def fn(data_source):
        data_source.append_to_list_field(path, [2])
    expected = [f'"{path}" does not specify a known field']
    _create_connect_and_check_assertion(initial_data={ 'a': [1] }, expected=expected, fn=fn)

def append_fails_due_to_not_a_list_field():
    path = 'a'
    def fn(data_source):
        data_source.append_to_list_field(path, [2])
    expected = [f'"{path}" does not specify a list field']
    _create_connect_and_check_assertion(initial_data={ 'a': 1 }, expected=expected, fn=fn)

def append_fails_due_to_not_a_list_value():
    def fn(data_source):
        data_source.append_to_list_field('a', 'abc')
    expected = ['values is not a list']
    _create_connect_and_check_assertion(initial_data={ 'a': [1] }, expected=expected, fn=fn)

def append_fails_due_to_large_message():
    path, values = 'a', ['abc'.join(['def'] * 4444)]
    expected = [f'delta is too large for "{path}"']
    def fn(data_source):
        data_source.append_to_list_field(path, values)
    _create_connect_and_check_assertion(initial_data={ path: [1] }, expected=expected, fn=fn, error_type=RuntimeError)

def append_fails_due_to_large_update():
    path, values = 'a', ['abc'.join(['def'] * 2222)]
    expected = [f'data object is too large']
    def fn(data_source):
        data_source.append_to_list_field(path, values)
    _create_connect_and_check_assertion(initial_data={ path: ['abc'.join(['def'] * 16600)] }, expected=expected, fn=fn, error_type=RuntimeError)

def remove_simple_succeeds():
    path = 'a'
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', { 'debug': { 'delta': [[path]] }}]
    def fn(data_source):
        data_source.remove_field(path)
    _create_and_connect({ path: [1] }, fn, expected)

def remove_complex_succeeds():
    path = 'foo.bar[0][1].baz'
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', { 'debug': { 'delta': [[path]] }}]
    def fn(data_source):
        data_source.remove_field(path)
    _create_and_connect({ 'foo': { 'bar': [[{ 'baz': 1 }, { 'baz': 2 }]] } }, fn, expected)

def different_removes_succeed():
    path1, path2 = 'a', 'b'
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', { 'debug': { 'delta': [[path1], [path2]] }}]
    def fn(data_source):
        data_source.remove_field(path1)
        data_source.remove_field(path2)
    _create_and_connect({ path1: 1, path2: 2 }, fn, expected)

def first_remove_succeeds_and_second_remove_fails():
    path = 'a'
    expected = [
        '[DataSource.remove_field] ignoring removal of an unknown field "%s"',
        path,
        '[DataSource.on_message] unexpected response from server:  "%s"',
        {'debug': { 'delta': [[path]]}},
    ]
    def fn(data_source):
        data_source.remove_field(path)
        expected = [f'"{path}" does not specify a known field']
        try:
            data_source.remove_field(path)
        except AssertionError as ex:
            actual = list(ex.args)
            if actual != expected:
                print('actual:', len(actual), *actual)
                print('expected:', len(expected), *expected)
        except BaseException as ex:
            print(f'actual {type(ex)}:', len(ex.args), *ex.args)
            print('expected AssertionError:', len(expected), *expected)
    _create_and_connect({ path: [1] }, fn, expected)

def remove_then_update_succeeds():
    path, value = 'a', 2
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', { 'debug': { 'delta': [[path, value]] }}]
    def fn(data_source):
        data_source.remove_field(path)
        data_source.update_field(path, value)
    _create_and_connect({ path: 1 }, fn, expected)

def remove_fails_due_to_no_connection():
    path = 'a'
    expected = ['connection not established']
    configuration = _create_configuration(initial_data={ path: [1] })
    data_source = DataSource()
    try:
        data_source.remove_field(path)
        print('unexpected success')
    except AssertionError as ex:
        actual = list(ex.args)
        if actual != expected:
            print('actual:', len(actual), *actual)
            print('expected:', len(expected), *expected)
    except BaseException as ex:
        print(f'actual {type(ex)}:', len(ex.args), *ex.args)
        print('expected AssertionError:', len(expected), *expected)

def remove_fails_due_to_empty_path():
    def fn(data_source):
        data_source.remove_field('')
    _create_connect_and_check_assertion(initial_data={ 'a': 1 }, expected=['path is empty'], fn=fn)

def remove_fails_due_to_invalid_path_type():
    def fn(data_source):
        data_source.remove_field(1)
    _create_connect_and_check_assertion(initial_data={ 'a': 1 }, expected=['path is not a string'], fn=fn)

def remove_fails_due_to_list_reference():
    path = 'a[0]'
    def fn(data_source):
        data_source.remove_field(path)
    _create_connect_and_check_assertion(initial_data={ 'a': [1] }, expected=[f'"{path}" does not specify a field'], fn=fn)

def remove_succeeds_with_metadata_reference():
    path = '_metadata'
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', { 'debug': { 'delta': [[path]] }}]
    def fn(data_source):
        data_source.remove_field(path)
    _create_and_connect({ path: { 'a': 1 } }, fn, expected)

def remove_fails_due_to_invalid_path():
    path = 'a[0'
    def fn(data_source):
        data_source.remove_field(path)
    expected = [f'"{path}" is not a valid field specifier']
    _create_connect_and_check_assertion(initial_data={ 'a': 1 }, expected=expected, fn=fn)

def remove_fails_due_to_no_field():
    path = 'b'
    def fn(data_source):
        data_source.remove_field(path)
    expected = [f'"{path}" does not specify a known field']
    _create_connect_and_check_assertion(initial_data={ 'a': 1 }, expected=expected, fn=fn)

def update_simple_succeeds():
    path, value = 'a', 1
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', { 'debug': { 'delta': [[path, value]] }}]
    def fn(data_source):
        data_source.update_field(path, value)
    _create_and_connect(None, fn, expected)

def update_complex_succeeds():
    path, value = 'foo.bar[0][1].baz', 3
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', { 'debug': { 'delta': [[path, value]] }}]
    def fn(data_source):
        data_source.update_field(path, value)
    _create_and_connect({ 'foo': { 'bar': [[{ 'baz': 1 }, { 'baz': 2 }]] } }, fn, expected)

def update_array_succeeds():
    path, value = 'foo.bar[0]', 3
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', { 'debug': { 'delta': [[path, value]] }}]
    def fn(data_source):
        data_source.update_field(path, value)
    _create_and_connect({ 'foo': { 'bar': [[{ 'baz': 1 }, { 'baz': 2 }]] } }, fn, expected)

def double_update_succeeds():
    path, value = 'a', 2
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', { 'debug': { 'delta': [[path, value]] }}]
    def fn(data_source):
        data_source.update_field(path, 1)
        data_source.update_field(path, value)
    _create_and_connect(None, fn, expected)

def different_updates_succeed():
    path1, value1 = 'a', 1
    path2, value2 = 'b', 2
    expected = ['[DataSource.on_message] unexpected response from server:  "%s"', { 'debug': { 'delta': [[path1, value1], [path2, value2]] }}]
    def fn(data_source):
        data_source.update_field(path1, value1)
        data_source.update_field(path2, value2)
    _create_and_connect(None, fn, expected)

def update_fails_due_to_no_connection():
    expected = ['connection not established']
    configuration = _create_configuration(initial_data={ 'a': 1 })
    data_source = DataSource()
    try:
        data_source.update_field('a', 2)
        print('unexpected success')
    except AssertionError as ex:
        actual = list(ex.args)
        if actual != expected:
            print('actual:', len(actual), *actual)
            print('expected:', len(expected), *expected)
    except BaseException as ex:
        print(f'actual {type(ex)}:', len(ex.args), *ex.args)
        print('expected AssertionError:', len(expected), *expected)

def update_fails_due_to_large_message():
    path, value = 'a', 'abc'.join(['def'] * 4444)
    expected = [f'delta is too large for "{path}"']
    def fn(data_source):
        data_source.update_field(path, value)
    _create_connect_and_check_assertion(initial_data={ 'a': 1 }, expected=expected, fn=fn, error_type=RuntimeError)

def update_fails_due_to_large_update():
    path, value = 'b', 'abc'.join(['def'] * 2222)
    expected = [f'data object is too large']
    def fn(data_source):
        data_source.update_field(path, value)
    _create_connect_and_check_assertion(initial_data={ 'a': 'abc'.join(['def'] * 16600) }, expected=expected, fn=fn, error_type=RuntimeError)

def update_fails_due_to_empty_path():
    def fn(data_source):
        data_source.update_field('', 2)
    _create_connect_and_check_assertion(initial_data={ 'a': 1 }, expected=['path is empty'], fn=fn)

def update_fails_due_to_invalid_path_type():
    def fn(data_source):
        data_source.update_field(1, 2)
    _create_connect_and_check_assertion(initial_data={ 'a': 1 }, expected=['path is not a string'], fn=fn)

def update_fails_due_to_invalid_metadata():
    def fn(data_source):
        data_source.update_field('_metadata', 1)
    _create_connect_and_check_assertion(initial_data={ 'a': 1 }, expected=['_metadata field is not a dictionary'], fn=fn)

def update_fails_due_to_invalid_path():
    pass
    path = 'a[0'
    def fn(data_source):
        data_source.update_field(path, 2)
    expected = [f'"{path}" is not a valid field specifier']
    _create_connect_and_check_assertion(initial_data={ 'a': 1 }, expected=expected, fn=fn)

def update_fails_due_to_no_field():
    path = 'b'
    def fn(data_source):
        data_source.update_field(path, 2)
    expected = [f'"{path}" does not specify a known field']
    _create_connect_and_check_assertion(initial_data={ 'a': 1 }, expected=expected, fn=fn)

def update_fails_due_to_out_of_bounds():
    path = 'a[1]'
    def fn(data_source):
        data_source.update_field(path, 2)
    expected = [f'"{path}" is out of bounds']
    _create_connect_and_check_assertion(initial_data={ 'a': [1] }, expected=expected, fn=fn)
