import json
import re
from functools import reduce
from random import randrange
from typing import Any, List, Optional, Union

def create_connect_message(session_id: str, token: str, broadcaster_ids: List[str], game_id: str, environment: str, is_debug: bool, data: Any) -> dict:
    connect = {
        'session_id': session_id,
        'token': token,
        'game_id': game_id,
        'env': environment,
        'data': data,
    }
    if broadcaster_ids:
        connect['broadcaster_ids'] = broadcaster_ids
    if is_debug:
        connect['debug'] = True
    return { 'connect': connect }

segment_rx = re.compile(r'[.[]')

def check_segment(data: Union[dict, list], segment: str) -> Union[None, dict, list]:
    if type(data) is dict:
        return data[segment]
    if type(data) is list and segment.endswith(']'):
        index = int(segment[:-1])
        return data[index]

def get_segment(data: dict, path: str) -> Optional[dict]:
    parts = segment_rx.split(path)
    field = parts.pop()
    if field:
        parent = reduce(check_segment, parts, data)
        if parent is not None:
            if field.endswith(']'):
                if type(parent) is list:
                    index = int(field[:-1])
                    return { 'parent': parent, 'field': index }
            elif type(parent) is dict:
                return { 'parent': parent, 'field': field }

path_rx = re.compile(r"^\w+(\.\w+|\[[0-9]\])*$")

def validate_path(path: str) -> str:
    if not path:
        raise AssertionError('path is empty')
    elif type(path) is not str:
        raise AssertionError('path is not a string')
    elif not path_rx.match(path):
        raise AssertionError(f'"{path}" is not a valid field specifier')
    return path

id_characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz'

def generate_session_id() -> str:
    return ''.join(id_characters[randrange(0, len(id_characters))] for _ in range(32))

def validate_data(data: Any) -> dict:
    if type(data) is not dict:
        raise AssertionError('data is not a dictionary')

    # If there is a "_metadata" field, ensure it is an object.
    metadata = data.get('_metadata')
    if metadata is not None and type(metadata) is not dict:
        raise AssertionError('_metadata field is not a dictionary')

    return json.loads(json.dumps(data))
