#include "pch.h"
#include "../../base-sdk/Shared/Http.h"
#include "../Shared/Twitch.h"
#include "../../base-sdk/Shared/Json.h"

using namespace std::literals;
using namespace Twitch;

using data_t = Json::object;

#if defined(__NX__) || defined(__ORBIS__)
constexpr std::chrono::seconds Twitch::DataSource::Configuration::DefaultTimeout /*= std::chrono::seconds(10)*/;
#endif

namespace {
	std::basic_regex<tstring::value_type> pathRx(_T("^\\w+(\\.\\w+|\\[[0-9]\\])*$"), std::regex_constants::ECMAScript);

	enum class DeltaOperationType { Append, Reauthorize, Reconnect, Refresh, Remove, Update };

	tstring GenerateSessionId() {
		tstring sessionId;
		std::generate_n(std::back_inserter(sessionId), 32, [] {
			unsigned int value;
			rand_s(&value);
			return static_cast<tstring::value_type>(((value & 0x6f) | 0x40) + 1);
		});
		return sessionId;
	}

	void ValidateData(data_t const& data) {
		// If there is a "_metadata" field, ensure it is an object.
		auto const it = data.find("_metadata");
		if(it != data.cend() && !it->second.is_object()) {
			DebugWriteLine(_T("[ValidateData] _metadata field is not an object"));
			throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
		}
	}

	std::string const accessors = ".[";

	Json UpdateData(Json data, std::string const& path, std::function<Json(Json, std::string const&)> fn) {
		auto const i = path.find_first_of(accessors, 1);
		if(i == path.npos) {
			return fn(data, path);
		} else if(path.front() == '[') {
			// This is an array field access.
			if(!data.is_array()) {
				DebugWriteLine(_T("[UpdateData] \"%s\" is not an array"), ToTstring(path).c_str());
				throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
			}
			auto array = data.array_items();
			auto const index = std::stoul(path.substr(1));
			if(index >= array.size()) {
				DebugWriteLine(_T("[UpdateData] array index %d is out of bounds (%d) for \"%s\""),
					index, array.size(), ToTstring(path).c_str());
				throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
			}
			auto const rest = path.substr(path.find(']') + (path[i] == '.' ? 2 : 1));
			array[index] = UpdateData(array[index], rest, fn);
			return array;
		} else {
			// This is an object field access.
			if(!data.is_object()) {
				DebugWriteLine(_T("[UpdateData] \"%s\" is not an object"), ToTstring(path).c_str());
				throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
			}
			auto object = data.object_items();
			auto const first = path.substr(0, i);
			auto const it = object.find(first);
			if(it == object.cend()) {
				DebugWriteLine(_T("[UpdateData] \"%s\" does not specify a known field"), ToTstring(path).c_str());
				throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
			}
			auto const rest = path.substr(i + (path[i] == '.' ? 1 : 0));
			it->second = UpdateData(it->second, rest, fn);
			return object;
		}
	}
}

tstring const DataSource::EmptyData = _T("{}");

#include "DataSourceImpl.inl"

DataSource::DataSource() = default;
DataSource::DataSource(DataSource&&) noexcept = default;
DataSource& DataSource::operator=(DataSource&&) noexcept = default;
DataSource::~DataSource() = default;

tstring DataSource::Connect(Configuration const& configuration) {
	// Ensure Connect has not already been invoked.
	if(pimpl && pimpl->IsValid) {
		DebugWriteLine(_T("[DataSource::Connect] this already connected"));
		throw TwitchException(FromPlatformError(ERROR_INVALID_STATE));
	}
	tstring sessionId;
	pimpl = std::make_unique<DataSourceImpl>(configuration, sessionId);
	return sessionId;
}

void DataSource::Disconnect() {
	pimpl = nullptr;
}

void DataSource::RemoveField(tstring const& path) {
	// Ensure Connect has already been invoked.
	if(!pimpl || !pimpl->IsValid) {
		DebugWriteLine(_T("[DataSource::RemoveField] not connected"));
		throw TwitchException(FromPlatformError(ERROR_INVALID_STATE));
	}

	// Validate the path.
	if(path.empty()) {
		DebugWriteLine(_T("[DataSource::RemoveField] path is empty"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	} else if(path.back() == ']') {
		DebugWriteLine(_T("[DataSource::RemoveField] \"%s\" does not specify a field"), path.c_str());
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	} else if(!std::regex_search(path, pathRx)) {
		DebugWriteLine(_T("[DataSource::RemoveField] \"%s\" is not a valid field specifier"), path.c_str());
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}

	// Create a "Remove Field" message.
	auto const message = DeltaOperation(DeltaOperationType::Remove, path, false);

	// Update the current data.
	pimpl->Access([&message](auto& currentData, auto& queue) {
		currentData = UpdateData(currentData, message.path, [](Json data, std::string const& path) {
			auto object = data.object_items();
			if(!object.erase(path)) {
				DebugWriteLine(_T("[DataSource::RemoveField] \"%s\" is not a valid field specifier"), ToTstring(path).c_str());
				throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
			}
			return object;
		}).object_items();

		// Enqueue the "Remove Field" message.
		queue.push_back(message);
	});
}

void DataSource::UpdateFieldWithJson(tstring const& path, std::string const& value_) {
	// Ensure Connect has already been invoked.
	if(!pimpl || !pimpl->IsValid) {
		DebugWriteLine(_T("[DataSource::UpdateFieldWithJson] not connected"));
		throw TwitchException(FromPlatformError(ERROR_INVALID_STATE));
	}

	// Validate the path and value.
	if(path.empty()) {
		DebugWriteLine(_T("[DataSource::UpdateFieldWithJson] path is empty"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	} else if(!std::regex_search(path, pathRx)) {
		DebugWriteLine(_T("[DataSource::RemoveField] \"%s\" is not a valid field specifier"), path.c_str());
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}
	std::string parseError;
	auto value = Json::Parse(value_, parseError);
	if(!parseError.empty()) {
		DebugWriteLine(_T("[DataSource::UpdateFieldWithJson] JSON parse error:  %s"), ToTstring(parseError).c_str());
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	} else if(path == _T("_metadata") && !value.is_object()) {
		DebugWriteLine(_T("[DataSource::UpdateFieldWithJson] _metadata update value is not a JSON object"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}

	// If updating the _metadata field, ensure it is valid.
	if(path == _T("_metadata")) {
		ValidateData({ { FromTstring(path), value } });
	}

	// Create an "Update Field" message.
	auto const message = DeltaOperation(DeltaOperationType::Update, path, value);

	pimpl->Access([&pimpl = pimpl, &message, &value](auto& currentData, auto& queue) {
		// Create an updated data object.
		auto newData = UpdateData(currentData, message.path, [&value](Json data, std::string const& path) -> Json {
			if(path.front() == '[') {
				// This is an array field access.
				auto array = data.array_items();
				auto const index = std::stoul(path.substr(1));
				if(index >= array.size()) {
					DebugWriteLine(_T("[DataSource::UpdateFieldWithJson] array index %d is out of bounds (%d) for \"%s\""),
						index, array.size(), ToTstring(path).c_str());
					throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
				}
				array[index] = value;
				return array;
			} else {
				if(!data.is_object()) {
					DebugWriteLine(_T("[DataSource::UpdateFieldWithJson] \"%s\" does not specify an object field"),
						ToTstring(path).c_str());
					throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
				}
				auto object = data.object_items();
				object[path] = value;
				return object;
			}
		}).object_items();

		// Ensure the updated data object is not too large.
		pimpl->CheckConnectMessageSize(newData);

		// Update the current data.
		currentData = newData;

		// Enqueue the "Update Field" message.
		queue.push_back(message);
	});
}

void DataSource::AppendToArrayFieldWithJson(tstring const& path, std::string const& value_) {
	// Ensure Connect has already been invoked.
	if(!pimpl || !pimpl->IsValid) {
		DebugWriteLine(_T("[DataSource::AppendToArrayFieldWithJson] not connected"));
		throw TwitchException(FromPlatformError(ERROR_INVALID_STATE));
	}

	// Validate the path and value.
	if(path.empty()) {
		DebugWriteLine(_T("[DataSource::AppendToArrayFieldWithJson] path is empty"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}
	std::string parseError;
	auto const value = Json::Parse(value_, parseError);
	if(!parseError.empty()) {
		DebugWriteLine(_T("[DataSource::AppendToArrayFieldWithJson] JSON parse error:  %s"), ToTstring(parseError).c_str());
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	} else if(!value.is_array()) {
		DebugWriteLine(_T("[DataSource::AppendToArrayFieldWithJson] value is not an array"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}
	auto const array = value.array_items();
	if(array.empty()) {
		DebugWriteLine(_T("[DataSource::AppendToArrayFieldWithJson] warning:  array is empty; ignoring"));
		return;
	}

	// Create an "Append To Array Field" message.
	auto const message = DeltaOperation(DeltaOperationType::Append, path, value);

	pimpl->Access([&pimpl = pimpl, &message, &array](auto& currentData, auto& queue) {
		// Create an updated data object.
		auto newData = UpdateData(currentData, message.path, [&array](Json data, std::string const& path)->Json {
			if(path.front() == '[') {
				// This is an array field access.
				auto parentArray = data.array_items();
				auto const index = std::stoul(path.substr(1));
				if(index >= parentArray.size()) {
					DebugWriteLine(_T("[DataSource::AppendToArrayFieldWithJson] array index %d is out of bounds (%d)"),
						index, parentArray.size());
					throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
				}
				auto& child = parentArray[index];
				if(!child.is_array()) {
					DebugWriteLine(_T("[DataSource::AppendToArrayFieldWithJson] current field value is not an array"));
					throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
				}
				auto childArray = child.array_items();
				childArray.insert(childArray.cend(), array.cbegin(), array.cend());
				child = childArray;
				return parentArray;
			} else {
				auto parentObject = data.object_items();
				auto& child = parentObject[path];
				if(!child.is_array()) {
					DebugWriteLine(_T("[DataSource::AppendToArrayFieldWithJson] current field value is not an array"));
					throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
				}
				auto childArray = child.array_items();
				childArray.insert(childArray.cend(), array.cbegin(), array.cend());
				child = childArray;
				return parentObject;
			}
		}).object_items();

		// Ensure the updated data object is not too large.
		pimpl->CheckConnectMessageSize(newData);

		// Update the current data.
		currentData = newData;

		// Enqueue the "Append To Array Field" message.
		queue.push_back(message);
	});
}
