#ifndef _WIN32
constexpr int ERROR_NOT_ENOUGH_SERVER_MEMORY = E2BIG;
#endif

namespace {
	constexpr size_t maximumConnectMessageSize = 100'000;
	constexpr size_t maximumDeltaSize = 20'000;
#ifndef _DEBUG
	constexpr
#endif
		std::chrono::milliseconds sendDelay = 1s;
#ifndef _DEBUG
	constexpr
#endif
		string_t url = _T("wss://metadata.twitch.tv/api/ingest");

	struct DeltaOperation {
		DeltaOperationType type;
		std::string path;
		Json value;

#ifdef _UNICODE
		DeltaOperation(DeltaOperationType type, tstring const& path, Json value) : DeltaOperation(type, FromTstring(path), value) {}
#endif
		DeltaOperation(DeltaOperationType type, std::string const& path, Json value);

		Json to_json() const {
			Json::array array;
			switch(type) {
			case DeltaOperationType::Append:
				array.push_back(path);
				array.push_back("a");
				array.push_back(value);
				break;
			case DeltaOperationType::Remove:
				array.push_back(path);
				break;
			case DeltaOperationType::Update:
				array.push_back(path);
				array.push_back(value);
				break;
			default:
				assert(false);
			}
			return array;
		}
	};

	struct Delta {
		std::vector<DeltaOperation> messages;

		Json to_json() const {
			Json::object rv;
			if(messages.front().type == DeltaOperationType::Refresh) {
				Json::object data{ { "data", messages.front().value } };
				rv.insert({ "refresh", data });
			} else {
				Json::array deltas;
				std::copy(messages.cbegin(), messages.cend(), std::back_inserter(deltas));
				rv.insert({ "delta", deltas });
			}
			return rv;
		}
	};

	DeltaOperation::DeltaOperation(DeltaOperationType type, std::string const& path, Json value) : type(type), path(path), value(value) {
		if(type == DeltaOperationType::Append || type == DeltaOperationType::Remove || type == DeltaOperationType::Update) {
			// Check if the message is too large.
			Delta delta{ { *this } };
			auto const s = delta.to_json().dump();
			if(s.size() > maximumDeltaSize) {
				DebugWriteLine(_T("[DeltaOperation::DeltaOperation] delta is too large for \"%s\""),
					(path.empty() ? std::to_string(static_cast<int>(type)) : path).c_str());
				throw TwitchException(FromPlatformError(ERROR_NOT_ENOUGH_SERVER_MEMORY));
			}
		}
	}

	class SynchronizedData {
	public:
		SynchronizedData() = default;
		SynchronizedData(SynchronizedData const&) = delete;
		SynchronizedData(SynchronizedData&&) = delete;
		SynchronizedData& operator=(SynchronizedData const&) = delete;
		SynchronizedData& operator=(SynchronizedData&&) = delete;
		~SynchronizedData() = default;

		template<typename FN>
		void Access(FN fn) {
			std::unique_lock<decltype(mutex)> lock(mutex);
			fn(currentData, queue);
		}

	private:
		data_t currentData;
		std::mutex mutex;
		std::deque<DeltaOperation> queue;
	};
}

#ifdef _DEBUG
namespace Twitch {
	void SetServiceParameters(std::chrono::milliseconds sendDelay_, string_t url_) {
		sendDelay = sendDelay_;
		url = url_;
	}
}
#endif

struct DataSource::DataSourceImpl {
	using token_t = decltype(Configuration::token);

	struct ConnectMessage {
		tstring sessionId;
		token_t token;
		std::vector<tstring> broadcasterIds;
		tstring gameId;
		tstring environment;
		bool isDebug;
		Json data;

		Json to_json() const {
			Json::object map;
			map.insert({ "session_id", FromTstring(sessionId) });
			map.insert({ "token", FromTstring(token) });
			if(!broadcasterIds.empty()) {
#ifdef _UNICODE
				std::vector<std::string> broadcasterIds_;
				std::transform(broadcasterIds.cbegin(), broadcasterIds.cend(), std::back_inserter(broadcasterIds_), FromTstring);
#else
				auto const& broadcasterIds_ = broadcasterIds;
#endif
				map.insert({ "broadcaster_ids", broadcasterIds_ });
			}
			map.insert({ "game_id", FromTstring(gameId) });
			map.insert({ "env", FromTstring(environment) });
			map.insert({ "data", data });
			if(isDebug) {
				map.insert({ "debug", true });
			}
			return data_t{ { "connect", map } };
		}
	};

	DataSourceImpl(Configuration const& configuration, tstring& sessionId) {
		// Validate the configuration.
		std::string parseError;
		auto const json = Json::Parse(FromTstring(configuration.initialData), parseError);
		if(!parseError.empty()) {
			DebugWriteLine(_T("[DataSourceImpl::DataSourceImpl] JSON parse error:  %s"), ToTstring(parseError).c_str());
			throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
		}
		if(!json.is_object()) {
			DebugWriteLine(_T("[DataSourceImpl::DataSourceImpl] initial data is not a JSON object"));
			throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
		}
		auto const data = json.object_items();
		ValidateData(data);
		onTokenExpired = configuration.onTokenExpired;
		token = configuration.token;
		sessionId = configuration.sessionId;
		if(sessionId.empty()) {
			sessionId = GenerateSessionId();
		}
		auto const isDebug = configuration.isDebug;

		// Set the data.
		synchronizedData.Access([&data](auto& currentData, auto&) {
			currentData = data;
		});

		// Ensure the initial connection message is not too large.
		createConnectMessageFn = [this, sessionId, broadcasterIds = configuration.broadcasterIds, gameId = configuration.gameId, environment = configuration.environment, isDebug](data_t const& data) {
			ConnectMessage connectMessage{ sessionId, token, broadcasterIds, gameId, environment, isDebug, data };
			return Json(connectMessage).dump();
		};
		auto const connectMessageSize = createConnectMessageFn(data).size();
		if(connectMessageSize > maximumConnectMessageSize) {
			DebugWriteLine(_T("[DataSourceImpl::DataSourceImpl] initial data object is too large"));
			throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
		}

		// Compose the connect function.
		connectFn = [this, sessionId, broadcasterIds = configuration.broadcasterIds, gameId = configuration.gameId, environment = configuration.environment](data_t const& data) {
			// Send the "connect" message.
			auto const messageText = createConnectMessageFn(data);
			return webSocket.Send(messageText);
		};

		// Compose the open function.
		openFn = [this, timeout = configuration.timeout]() {
			webSocket.OnClosed = [this] { EnqueueReconnect(); };
			webSocket.OnReceived = [this](std::string_view data) { ProcessMessage(data); };
			return webSocket.Open(url, timeout);
		};

		// Open the WebSocket.
		ThrowIfFailed(openFn());

		// Enqueue a "reauthorize" message without clearing the token.
		Enqueue({ DeltaOperationType::Reauthorize, "", false });

		// Start the sender task.
		auto future = closePromise.get_future().share();
		senderTask = std::async(std::launch::async, [this, future] { SendMessages(future); });
	}

	DataSourceImpl() = delete;
	DataSourceImpl(DataSourceImpl const&) = delete;
	DataSourceImpl(DataSourceImpl&&) = delete;
	DataSourceImpl& operator=(DataSourceImpl const&) = delete;
	DataSourceImpl& operator=(DataSourceImpl&&) = delete;

	~DataSourceImpl() {
		closePromise.set_value();
		senderTask.get();
	}

	bool GetIsValid() const { return webSocket.IsOpen; }
	__declspec(property(get = GetIsValid)) bool const IsValid;

	template<typename FN>
	void Access(FN fn) {
		synchronizedData.Access(fn);
	}

	void CheckConnectMessageSize(data_t const& data) {
		auto const s = createConnectMessageFn(data);
		if(s.size() > maximumConnectMessageSize) {
			DebugWriteLine(_T("[DataSourceImpl::CheckConnectMessageSize] data object is too large"));
			throw TwitchException(FromPlatformError(ERROR_NOT_ENOUGH_SERVER_MEMORY));
		}
	}

	void Enqueue(DeltaOperation const& message) {
#ifdef _DEBUG
		if(!webSocket.IsOpen && message.type != DeltaOperationType::Reconnect) {
			DebugWriteLine(_T("[DataSourceImpl::Enqueue] not connected"));
			assert(webSocket.IsOpen);
		}
#endif
		Access([&message](auto&, auto& queue) {
			queue.push_back(message);
		});
	}

private:
	SynchronizedData synchronizedData;
	WebSocket webSocket;
	std::function<std::string(data_t const&)> createConnectMessageFn;
	std::function<int(data_t const&)> connectFn;
	std::function<int()> openFn;
	std::future<void> senderTask;
	std::promise<void> closePromise;
	token_t token;
	decltype(Configuration::onTokenExpired) onTokenExpired;
	static std::vector<DeltaOperationType> const uniqueMessageTypes /*= { DeltaOperationType::Reconnect, DeltaOperationType::Reauthorize, DeltaOperationType::Refresh }*/;

	token_t AcquireToken() {
		std::promise<token_t> promise;
		auto const fn = [&promise](token_t const& token) {
			promise.set_value(token);
		};
		try {
			if(!onTokenExpired) {
				DebugWriteLine(_T("[DataSourceImpl::AcquireToken] warning:  onTokenExpired not provided"));
				return token_t();
			}
			if(onTokenExpired(fn)) {
				// The client indicated it will fetch a token and invoke the call-back.
				return promise.get_future().get();
			} else {
				// The client wants to shut down.
				return token_t();
			}
		} catch(...) {
			// There is likely a logic error in the client.  Shut down the connection.
			DebugWriteLine(_T("[DataSourceImpl::AcquireToken] onTokenExpired threw an exception"));
			return token_t();
		}
	}

	void SendMessages(std::shared_future<void> future) {
		auto clock = std::chrono::steady_clock();
		auto sendTime = clock.now();
		decltype(sendDelay - (sendTime - sendTime)) adjustedSendDelay;
		do {
#ifdef __NX__
			// Perform Web socket processing.
			webSocket.Process();
#endif
			// Check the queue for unique modifications.
			data_t data;
			Delta delta;
			Access([&data, &delta](auto& currentData, auto& queue) {
				data = currentData;
				for(auto const uniqueMessageType : uniqueMessageTypes) {
					auto const it = std::find_if(queue.cbegin(), queue.cend(), [uniqueMessageType](auto const& message) { return message.type == uniqueMessageType; });
					if(it != queue.cend()) {
						// Found one; use it.
						delta.messages.push_back(*it);
						queue.clear();
						return;
					}
				}

				// Trim the queue based on repeated modifications.
				if(!queue.empty()) {
					std::remove_reference_t<decltype(queue)> trimmedQueue;
					while(!queue.empty()) {
						auto const message = queue.front();
						queue.pop_front();
						switch(message.type) {
						case DeltaOperationType::Append:
							ExtendAppend(trimmedQueue, message);
							break;
						case DeltaOperationType::Remove:
						case DeltaOperationType::Update:
							trimmedQueue.erase(std::remove_if(trimmedQueue.begin(), trimmedQueue.end(), [&message](auto const& trimmedMessage) {
								return trimmedMessage.path == message.path;
							}), trimmedQueue.end());
							trimmedQueue.push_back(message);
							break;
						default:
							assert(false);
						}
					}
					queue.swap(trimmedQueue);
				}

				// Take messages off of the queue until reaching the maximum payload size.
				while(!queue.empty()) {
					delta.messages.push_back(queue.front());
					if(delta.to_json().dump().size() > maximumDeltaSize) {
						delta.messages.pop_back();
						break;
					}
					queue.pop_front();
				}
			});

			// If messages were removed from the queue, send them.
			if(!delta.messages.empty()) {
				std::string messageText;
				switch(delta.messages.front().type) {
					int reconnectionDelay, errorCode;
				case DeltaOperationType::Reauthorize:
					if(delta.messages.front().value.bool_value()) {
						token.clear();
					}
					if(!Reauthorize(data)) {
						return;
					}
					break;
				case DeltaOperationType::Reconnect:
					// Close the current connection.
					CloseWebSocket();

					// Await the requested reconnection delay.
					reconnectionDelay = delta.messages.front().value.int_value() - GetTickCount();
					if(reconnectionDelay > 0) {
						if(future.wait_for(std::chrono::milliseconds(reconnectionDelay)) != std::future_status::timeout) {
							// The client is shutting down.
							return;
						}
					}

					// Open a new connection.
					errorCode = openFn();
					if(errorCode) {
						// Cannot re-establish a connection.
						assert(!webSocket.IsOpen);
						DebugWriteLine(_T("[DataSourceImpl::SendMessages] WebSocket::Open returned error %#x (%d)"),
							errorCode, errorCode & 0xffff);
						return;
					}

					// Perform the "Reauthorize" action above.
					if(!Reauthorize(data)) {
						return;
					}
					break;
				case DeltaOperationType::Refresh:
					delta.messages.front().value = data;
					__fallthrough;
				default:
					messageText = delta.to_json().dump();
				}
				if(!messageText.empty()) {
					int errorCode = webSocket.Send(messageText);
					if(errorCode) {
						if(webSocket.IsOpen) {
							// An error occurred.  Enqueue a "refresh" message and try again.
							DebugWriteLine(_T("[DataSourceImpl::SendMessages] WebSocket::Send returned error %#x (%d)"),
								errorCode, errorCode & 0xffff);
							Access([](auto&, auto& queue) {
								queue.clear();
								queue.push_back({ DeltaOperationType::Refresh, "", false });
							});
						} else {
							// Since there is no active connection, either the client wants to shut down or
							// the server either requested a reconnection or unexpectedly closed the
							// connection.  In the first case, the future will be set and this loop will
							// exit.  In the latter cases, the receive or close handler, respectively, will
							// post a "reconnect" message.  In any case, it's okay to lose the current message.
						}
#ifdef __NX__
					} else {
						// Perform Web socket processing.
						webSocket.Process();
#endif
					}
				}
			}
			// Increment the send time by one second and wait until that time before
			// sending the next message.
			sendTime += sendDelay;
			auto now = clock.now();
			adjustedSendDelay = sendTime - now;
			if(adjustedSendDelay.count() < 0) {
				adjustedSendDelay = decltype(adjustedSendDelay)();
			}
		} while(future.wait_for(adjustedSendDelay) == std::future_status::timeout);

		// Close the connection.
		CloseWebSocket();
	}

	bool Reauthorize(data_t const& data) {
		if(token.empty()) {
			token = AcquireToken();
		}
		if(!token.empty()) {
			auto const errorCode = connectFn(data);
			if(!errorCode) {
				return true;
			}
			DebugWriteLine(_T("[DataSourceImpl::Reauthorize] reconnection failed: error %#x (%d)"), errorCode, errorCode & 0xffff);
		}
		CloseWebSocket();
		return false;
	}

	void CloseWebSocket() {
		webSocket.OnClosed = WebSocket::DefaultClosedFn;
		webSocket.OnReceived = WebSocket::DefaultReceivedFn;
		webSocket.Close();
	}

	static void ExtendAppend(std::deque<DeltaOperation>& queue, DeltaOperation const& message) {
		auto const it = std::find_if(queue.begin(), queue.end(), [&message](auto const& trimmedMessage) {
			return trimmedMessage.type == message.type && trimmedMessage.path == message.path;
		});
		if(it == queue.end()) {
			queue.push_back(message);
		} else {
			assert(it->type == DeltaOperationType::Append);
			assert(it->value.is_array());
			auto currentArray = it->value.array_items();
			assert(message.value.is_array());
			auto const& array = message.value.array_items();
			currentArray.insert(currentArray.cend(), array.cbegin(), array.cend());
			it->value = currentArray;
		}
	}

	void EnqueueReconnect(unsigned reconnectTime = GetTickCount()) {
		// Clear the handlers and enqueue a "reconnect" message.
		webSocket.OnClosed = WebSocket::DefaultClosedFn;
		webSocket.OnReceived = WebSocket::DefaultReceivedFn;
		Enqueue({ DeltaOperationType::Reconnect, "", static_cast<int>(reconnectTime) });
	}

	void ProcessMessage(std::string_view responseData) {
		std::string parseError;
		auto const response = Json::Parse(responseData, parseError);
		if(!parseError.empty()) {
			DebugWriteLine(_T("[DataSourceImpl::ProcessMessage] invalid JSON from server:  %s"), ToTstring(parseError).c_str());
		}
		auto const error = response["error"];
		if(error.is_object()) {
			auto code = error["code"].string_value();
			if(code == "invalid_connect_token") {
				// The server has rejected the authorization token.
				Enqueue({ DeltaOperationType::Reauthorize, "", true });
			} else if(code == "connection_not_authed") {
				// Ignore this since it means we're in the middle of getting connected.
			} else if(code == "waiting_on_refresh_message") {
				// Ignore this since it means we're in the middle of refreshing.
			} else {
				auto const field = error["error_field"];
				if(field.is_string()) {
					code += ":  \"" + field.string_value() + '"';
				}
				DebugWriteLine(_T("[DataSourceImpl::ProcessMessage] unexpected error from server:  %s"), ToTstring(code).c_str());
			}
			return;
		}
		auto const reconnect = response["reconnect"];
		if(reconnect.is_number()) {
			// The server requested a reconnection.
			auto const reconnectTime = reconnect.int_value() + GetTickCount();
			EnqueueReconnect(reconnectTime);
			return;
		}
		if(!response["connected"].bool_value()) {
			DebugWriteLine(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  %s"),
				tstring(responseData.cbegin(), responseData.cend()).c_str());
		}
		DebugWriteLine(_T("[DataSourceImpl::ProcessMessage] connected"));
	}
};

std::vector<DeltaOperationType> const DataSource::DataSourceImpl::uniqueMessageTypes = { DeltaOperationType::Reconnect, DeltaOperationType::Reauthorize, DeltaOperationType::Refresh };
