// Copyright (c) 2020, Twitch Interactive, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
// OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// OTHER DEALINGS IN THE SOFTWARE.

#pragma once

#include <chrono>
#include <functional>
#include <memory>
#include <sstream>
#include <vector>

namespace Twitch {
	class DataSource;
}

class Twitch::DataSource {
public:
	using TokenRefreshFn = std::function<void(tstring const&)>;
	using TokenExpiredFn = std::function<bool(TokenRefreshFn)>;

	struct Configuration {
		Configuration() = default;
		Configuration(Configuration const&) = default;
		Configuration(Configuration&&) = default;
		Configuration& operator=(Configuration const&) = default;
		Configuration& operator=(Configuration&&) = default;
		~Configuration() = default;

		tstring sessionId;
		tstring token;
		std::vector<tstring> broadcasterIds;
		tstring gameId;
		tstring environment;
		TokenExpiredFn onTokenExpired;
		tstring initialData = EmptyData;
		std::chrono::milliseconds timeout = DefaultTimeout;
		bool isDebug;

		static constexpr std::chrono::seconds DefaultTimeout = std::chrono::seconds(10);
	};

	DataSource();
	DataSource(DataSource const&) = delete;
	DataSource(DataSource&&) noexcept;
	DataSource& operator=(DataSource const&) = delete;
	DataSource& operator=(DataSource&&) noexcept;
	~DataSource();

	tstring Connect(Configuration const& configuration);
	void Disconnect();
	void RemoveField(tstring const& path);
	void UpdateFieldWithJson(tstring const& path, std::string const& value);
	void AppendToArrayFieldWithJson(tstring const& path, std::string const& value);
	template<typename T>
	void UpdateField(tstring const& path, T const& value) {
		std::stringstream ss;
		Stringify(ss, value);
		UpdateFieldWithJson(path, ss.str());
	}
	template<typename T>
	void AppendToArrayField(tstring const& path, T const& value) {
		std::stringstream ss;
		Stringify(ss, value);
		AppendToArrayFieldWithJson(path, ss.str());
	}

	static tstring const EmptyData /*= _T("{}")*/;

private:
	struct DataSourceImpl;

	std::unique_ptr<DataSourceImpl> pimpl;
};
