import createDataSource, { Data, TokenRefreshFn } from '../lib/index';
import { createAndConnect, createConsolePromise, createConfiguration, createTokenExpiredPromise, readFile, replaceConsoleError } from './utilities';

const consoleError: Function = replaceConsoleError();

test('connect succeeds', async () => {
  const promise = createConsolePromise();
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  await dataSource.connect(configuration);
  expect(await promise).toBe('"connected"');
  await dataSource.disconnect();
});

test('connect with debugging succeeds', async () => {
  const promise = createConsolePromise();
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  configuration.debugFn = true;
  await dataSource.connect(configuration);
  expect(await promise).toBe('"connected"');
  await dataSource.disconnect();
});

test('connect with custom debugFn succeeds', async () => {
  const promise = createConsolePromise();
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  let debugFnInvoked;
  configuration.debugFn = (a: string, b: string, c: string) => {
    console.error(a, b, c);
    debugFnInvoked = true;
  }
  await dataSource.connect(configuration);
  expect(await promise).toBe('"connected"');
  expect(debugFnInvoked).toBe(true);
  await dataSource.disconnect();
});

test('connect with user token succeeds', async () => {
  const promise = createConsolePromise();
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  configuration.token = 'user';
  delete configuration.broadcasterIds;
  await dataSource.connect(configuration);
  expect(await promise).toBe('"connected"');
  await dataSource.disconnect();
});

test('connect with empty broadcaster ID array fails', async () => {
  const promise = createConsolePromise();
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  configuration.broadcasterIds = [];
  await dataSource.connect(configuration);
  expect(await promise).toBe('{"error":{"code":"connect_invalid_values","message":"connect message has an invalid value for a required field","error_field":"broadcaster_ids"}}');
  await dataSource.disconnect();
});

test('connect with no broadcaster ID array fails', async () => {
  const promise = createConsolePromise();
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  delete configuration.broadcasterIds;
  await dataSource.connect(configuration);
  expect(await promise).toBe('{"error":{"code":"connect_invalid_values","message":"connect message has an invalid value for a required field","error_field":"broadcaster_ids"}}');
  await dataSource.disconnect();
});

test('connect with immediate token acquisition succeeds', async () => {
  const promise = createConsolePromise();
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  configuration.token = '';
  configuration.onTokenExpired = (fn: TokenRefreshFn) => {
    fn('token');
    return true;
  };
  await dataSource.connect(configuration);
  expect(await promise).toBe('"connected"');
  await dataSource.disconnect();
});

test('connect with delayed token acquisition succeeds', async () => {
  const promise = createConsolePromise();
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  configuration.token = '';
  configuration.onTokenExpired = (fn: TokenRefreshFn) => {
    setTimeout(() => fn('token'), 11);
    return true;
  };
  await dataSource.connect(configuration);
  expect(await promise).toBe('"connected"');
  await dataSource.disconnect();
});

test('connect fails due to token', async () => {
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  configuration.token = 'bad-token';
  const promise = createTokenExpiredPromise(configuration);
  await dataSource.connect(configuration);
  await promise;
  await dataSource.disconnect();
});

test('connect fails due to current connection', async () => {
  expect.hasAssertions();
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  await dataSource.connect(configuration);
  try {
    await dataSource.connect(configuration);
    consoleError('unexpected success');
    await dataSource.disconnect();
  } catch (ex) {
    expect(ex.message).toBe('already connected');
  }
  await dataSource.disconnect();
});

test('connect fails due to not an object', async () => {
  expect.hasAssertions();
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  configuration.initialData = '$' as any as Data;
  try {
    await dataSource.connect(configuration);
    consoleError('unexpected success');
    await dataSource.disconnect();
  } catch (ex) {
    expect(ex.message).toBe('data is not an object');
  }
});

test('connect fails due to size', async () => {
  expect.hasAssertions();
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  configuration.initialData.large = Array(444).fill(Array(999).fill('a').join('b'));
  try {
    await dataSource.connect(configuration);
    consoleError('unexpected success');
    await dataSource.disconnect();
  } catch (ex) {
    expect(ex.message).toBe('initial data object is too large');
  }
});

test('connect fails due to invalid _metadata', async () => {
  expect.hasAssertions();
  const dataSource = createDataSource();
  const initialData = { _metadata: 1 };
  const configuration = createConfiguration(initialData);
  try {
    await dataSource.connect(configuration);
    consoleError('unexpected success');
    await dataSource.disconnect();
  } catch (ex) {
    expect(ex.message).toBe('_metadata field is not an object');
  }
});

test('connect fails due to invalid onTokenExpired', async () => {
  expect.hasAssertions();
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  configuration.onTokenExpired = 1 as any;
  try {
    await dataSource.connect(configuration);
    consoleError('unexpected success');
    await dataSource.disconnect();
  } catch (ex) {
    expect(ex.message).toBe('on_token_expired is not callable');
  }
});

test('connect fails due to onTokenExpired exception', async () => {
  expect.hasAssertions();
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  configuration.token = '';
  configuration.onTokenExpired = (_fn: TokenRefreshFn) => {
    throw new Error('test');
  };
  let actualMethod, actualMessage, actualError;
  const promise = new Promise<boolean>((resolve) => {
    const timerId = setTimeout(() => {
      resolve(false);
    }, 2222);
    configuration.debugFn = (method: string, message: string, error: Error) => {
      actualMethod = method;
      actualMessage = message;
      actualError = error && error.message;
      clearTimeout(timerId);
      resolve(true);
    }
  });
  await dataSource.connect(configuration);
  expect(await promise).toBe(true);
  expect(actualMethod).toBe('DataSource.acquireToken');
  expect(actualMessage).toBe('onTokenExpired threw an exception');
  expect(actualError).toBe('test');
  await dataSource.disconnect();
});

test('component handles invalid JSON from server', async () => {
  const promise = createConsolePromise();
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  configuration.token = 'unexpected-payload';
  configuration.gameId = '$';
  await dataSource.connect(configuration);
  expect(await promise).toBe('Unexpected token $ in JSON at position 0');
  await dataSource.disconnect();
});

test('component handles unexpected JSON from server', async () => {
  const promise = createConsolePromise();
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  configuration.token = 'unexpected-payload';
  configuration.gameId = '1';
  await dataSource.connect(configuration);
  expect(await promise).toBe('1');
  await dataSource.disconnect();
});

test('component aborts successfully', async () => {
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  configuration.token = 'abort';
  const promise = createConsolePromise();
  await dataSource.connect(configuration);
  expect(await promise).toBe('"connected"');
  await dataSource.disconnect();
});

test('component reconnects after long wait', async () => {
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  configuration.token = 'restart';
  configuration.gameId = '2200';
  const promise = createConsolePromise(4444);
  await dataSource.connect(configuration);
  expect(await promise).toBe('"connected"');
  await dataSource.disconnect();
});

test('component reconnects after short wait', async () => {
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  configuration.token = 'restart';
  configuration.gameId = '110';
  const promise = createConsolePromise();
  await dataSource.connect(configuration);
  expect(await promise).toBe('"connected"');
  await dataSource.disconnect();
});

test('component handles unexpected connect', async () => {
  const promise = createConsolePromise();
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  configuration.token = 'unexpected-payload';
  configuration.gameId = '{"connected":false}';
  await dataSource.connect(configuration);
  expect(await promise).toBe('{"connected":false}');
  await dataSource.disconnect();
});

test('component handles unexpected reconnect', async () => {
  const promise = createConsolePromise();
  const dataSource = createDataSource();
  const configuration = createConfiguration();
  configuration.token = 'unexpected-payload';
  configuration.gameId = '{"reconnect":null}';
  await dataSource.connect(configuration);
  expect(await promise).toBe('{"reconnect":null}');
  await dataSource.disconnect();
});

test('append succeeds', async () => {
  const dataSource = await createAndConnect({ a: [1] });
  const promise = createConsolePromise();
  const path = 'a';
  const value = [2];
  dataSource.appendToArrayField(path, value);
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path, 'a', value]] } }));
  await dataSource.disconnect();
});

test('append empty succeeds', async () => {
  const dataSource = await createAndConnect({ a: [1] });
  const promise = createConsolePromise();
  const path = 'a';
  const value: any[] = [];
  dataSource.appendToArrayField(path, value);
  expect(await promise).toBe('warning:  array is empty; ignoring');
  await dataSource.disconnect();
});

test('double-append succeeds', async () => {
  const dataSource = await createAndConnect({ a: [1] });
  const promise = createConsolePromise();
  const path = 'a';
  const value1 = [2];
  dataSource.appendToArrayField(path, value1);
  const value2 = [3];
  dataSource.appendToArrayField(path, value2);
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path, 'a', value1.concat(value2)]] } }));
  await dataSource.disconnect();
});

test('append then remove succeeds', async () => {
  const dataSource = await createAndConnect({ a: [1] });
  const promise = createConsolePromise();
  const path = 'a';
  dataSource.appendToArrayField(path, [2]);
  dataSource.removeField(path);
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path]] } }));
  await dataSource.disconnect();
});

test('append fails due to no connection', async () => {
  expect.hasAssertions();
  const dataSource = createDataSource();
  try {
    await dataSource.appendToArrayField('a', [1]);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe('connection not established');
  }
});

test('append fails due to empty path', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect({ a: [1] });
  try {
    await dataSource.appendToArrayField('', [2]);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe('path is empty');
  }
  await dataSource.disconnect();
});

test('append fails due to invalid path type', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect({ a: [1] });
  try {
    await dataSource.appendToArrayField(1 as any as string, [2]);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe('path is not a string');
  }
  await dataSource.disconnect();
});

test('append fails due to invalid path', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect({ a: [1] });
  const path = 'a[0';
  try {
    dataSource.appendToArrayField(path, [2]);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe(`"${path}" is not a valid field specifier`);
  }
  await dataSource.disconnect();
});

test('append fails due to no field', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect({ a: [1] });
  const path = 'b';
  try {
    dataSource.appendToArrayField(path, [2]);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe(`"${path}" does not specify a known field`);
  }
  await dataSource.disconnect();
});

test('append fails due to not an array field', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect({ a: 1 });
  const path = 'a';
  try {
    dataSource.appendToArrayField(path, [2]);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe(`"${path}" does not specify an array field`);
  }
  await dataSource.disconnect();
});

test('append fails due to not an array value', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect({ a: [1] });
  const path = 'a';
  try {
    dataSource.appendToArrayField(path, 2 as any as any[]);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe('values is not an array');
  }
  await dataSource.disconnect();
});

test('append fails due to large message', async () => {
  expect.hasAssertions();
  const path = 'a';
  const dataSource = await createAndConnect({ [path]: [1] });
  const values = Array(4444).fill('abcdef');
  try {
    dataSource.appendToArrayField(path, values);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe(`delta is too large for "${path}"`);
  }
  await dataSource.disconnect();
});

test('append fails due to large update', async () => {
  expect.hasAssertions();
  const path = 'a';
  const values = Array(1111).fill('abcdef');
  const dataSource = await createAndConnect({ [path]: Array(11000).fill('abcdef') });
  try {
    dataSource.appendToArrayField(path, values);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe('data object is too large');
  }
  await dataSource.disconnect();
});

test('remove simple succeeds', async () => {
  const path = 'a';
  const dataSource = await createAndConnect({ [path]: 1 });
  const promise = createConsolePromise();
  dataSource.removeField(path);
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path]] } }));
  await dataSource.disconnect();
});

test('remove complex succeeds', async () => {
  const dataSource = await createAndConnect({ foo: { bar: [[{ baz: 1 }, { baz: 2 }]] } });
  const promise = createConsolePromise();
  const path = 'foo.bar[0][1].baz';
  dataSource.removeField(path);
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path]] } }));
  await dataSource.disconnect();
});

test('different removes succeed', async () => {
  const path1 = 'a';
  const path2 = 'b';
  const dataSource = await createAndConnect({ [path1]: 1, [path2]: 2 });
  const promise = createConsolePromise();
  dataSource.removeField(path1);
  dataSource.removeField(path2);
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path1], [path2]] } }));
  await dataSource.disconnect();
});

test('first remove succeeds and second remove fails', async () => {
  const dataSource = await createAndConnect({ a: 1 });
  const promise = createConsolePromise();
  const path = 'a';
  dataSource.removeField(path);
  dataSource.removeField(path);
  expect(await promise).toBe('warning:  ignoring removal of an unknown field');
  await dataSource.disconnect();
});

test('remove then update succeeds', async () => {
  const path = 'a';
  const dataSource = await createAndConnect({ [path]: 1 });
  const promise = createConsolePromise();
  dataSource.removeField(path);
  const value = 2;
  dataSource.updateField(path, value);
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path, value]] } }));
  await dataSource.disconnect();
});

test('remove fails due to no connection', async () => {
  expect.hasAssertions();
  const dataSource = createDataSource();
  try {
    await dataSource.removeField('');
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe('connection not established');
  }
});

test('remove fails due to empty path', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect({ a: 1 });
  try {
    dataSource.removeField('');
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe('path is empty');
  }
  await dataSource.disconnect();
});

test('remove fails due to invalid path type', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect({ a: 1 });
  try {
    dataSource.removeField(1 as any as string);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe('path is not a string');
  }
  await dataSource.disconnect();
});

test('remove fails due to array reference', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect({ a: [1] });
  const path = 'a[0]';
  try {
    dataSource.removeField(path);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe(`"${path}" does not specify a field`);
  }
  await dataSource.disconnect();
});

test('remove succeeds with metadata reference', async () => {
  const dataSource = await createAndConnect({ _metadata: { a: 1 } });
  const promise = createConsolePromise();
  const path = '_metadata';
  dataSource.removeField(path);
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path]] } }));
  await dataSource.disconnect();
});

test('remove fails due to invalid path', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect();
  const path = 'a[0';
  try {
    dataSource.removeField(path);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe(`"${path}" is not a valid field specifier`);
  }
  await dataSource.disconnect();
});

test('remove fails due to no field', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect();
  const promise = createConsolePromise();
  const path = 'a';
  dataSource.removeField(path);
  expect(await promise).toBe('warning:  ignoring removal of an unknown field');
  await dataSource.disconnect();
});

test('update simple succeeds', async () => {
  const dataSource = await createAndConnect();
  const promise = createConsolePromise();
  const path = 'a';
  const value = 1;
  dataSource.updateField(path, value);
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path, value]] } }));
  await dataSource.disconnect();
});

test('update complex succeeds', async () => {
  const dataSource = await createAndConnect({ foo: { bar: [[{ baz: 1 }, { baz: 2 }]] } });
  const promise = createConsolePromise();
  const path = 'foo.bar[0][1].baz';
  const value = 3;
  dataSource.updateField(path, value);
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path, value]] } }));
  await dataSource.disconnect();
});

test('update array succeeds', async () => {
  const dataSource = await createAndConnect({ foo: { bar: [[{ baz: 1 }, { baz: 2 }]] } });
  const promise = createConsolePromise();
  const path = 'foo.bar[0]';
  const value = 3;
  dataSource.updateField(path, value);
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path, value]] } }));
  await dataSource.disconnect();
});

test('double-update succeeds', async () => {
  const dataSource = await createAndConnect();
  const promise = createConsolePromise();
  const path = 'a';
  dataSource.updateField(path, 1);
  const value = 2;
  dataSource.updateField(path, value);
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path, value]] } }));
  await dataSource.disconnect();
});

test('different updates succeed', async () => {
  const dataSource = await createAndConnect();
  const promise = createConsolePromise();
  const path1 = 'a';
  const value1 = 1;
  dataSource.updateField(path1, value1);
  const path2 = 'b';
  const value2 = 2;
  dataSource.updateField(path2, value2);
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path1, value1], [path2, value2]] } }));
  await dataSource.disconnect();
});

test('same update succeeds', async () => {
  const dataSource = await createAndConnect();
  const promise = createConsolePromise();
  const path = 'a';
  const value = 1;
  dataSource.updateField(path, value);
  dataSource.updateField(path, value);
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path, value]] } }));
  await dataSource.disconnect();
});

test('many updates succeed', async () => {
  const dataSource = await createAndConnect();
  const promise = createConsolePromise();
  const value = Array(99).fill('abcdef');
  const delta = [];
  for (let i = 0; i < 99; ++i) {
    const path = 'a' + i.toString();
    if (i < 22) {
      delta.push([path, value]);
    }
    dataSource.updateField(path, value);
  }
  expect(await promise).toBe(JSON.stringify({ debug: { delta } }));
  await dataSource.disconnect();
});

test('update fails due to no connection', async () => {
  expect.hasAssertions();
  const dataSource = createDataSource();
  try {
    await dataSource.updateField('', {});
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe('connection not established');
  }
});

test('update fails due to large message', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect({});
  const path = 'a';
  try {
    dataSource.updateField(path, Array(4444).fill('abc').join('def'));
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe(`delta is too large for "${path}"`);
  }
  await dataSource.disconnect();
});

test('update fails due to large update', async () => {
  expect.hasAssertions();
  const value = Array(1111).fill('abcdef');
  const dataSource = await createAndConnect({ 'a': Array(11000).fill('abcdef') });
  try {
    dataSource.updateField('b', value);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe('data object is too large');
  }
  await dataSource.disconnect();
});

test('update fails due to empty path', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect({ a: 1 });
  try {
    dataSource.updateField('', 2);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe('path is empty');
  }
  await dataSource.disconnect();
});

test('update fails due to invalid path type', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect({ a: 1 });
  try {
    dataSource.updateField(1 as any as string, 2);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe('path is not a string');
  }
  await dataSource.disconnect();
});

test('update fails due to invalid _metadata', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect({ a: 1 });
  try {
    dataSource.updateField('_metadata', 1);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe('_metadata field is not an object');
  }
  await dataSource.disconnect();
});

test('update fails due to invalid path', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect();
  const path = 'a[0';
  try {
    dataSource.updateField(path, 1);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe(`"${path}" is not a valid field specifier`);
  }
  await dataSource.disconnect();
});

test('update fails due to no field', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect();
  const path = 'a.b';
  try {
    dataSource.updateField(path, 1);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe(`"${path}" does not specify a known field`);
  }
  await dataSource.disconnect();
});

test('update fails due to out of bounds', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect({ a: [1] });
  const path = 'a[1]';
  try {
    dataSource.updateField(path, 1);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe(`"${path}" is out of bounds`);
  }
  await dataSource.disconnect();
});

test('auto-update add array element succeeds', async () => {
  const dataSource = await createAndConnect();
  const promise = createConsolePromise();
  const value = 5;
  dataSource.updateData({
    foo: {
      bar: [[
        {
          one: [1, 2]
        },
        {
          two: [3, 4, value]
        }
      ]]
    }
  });
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [['foo.bar[0][1].two', 'a', [value]]] } }));
  await dataSource.disconnect();
});

test('auto-update add two array elements succeeds', async () => {
  const dataSource = await createAndConnect();
  const promise = createConsolePromise();
  const value = 5;
  dataSource.updateData({
    foo: {
      bar: [[
        {
          one: [1, 2]
        },
        {
          two: [3, 4, value, value]
        }
      ]]
    }
  });
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [['foo.bar[0][1].two', 'a', [value, value]]] } }));
  await dataSource.disconnect();
});

test('auto-update fails due to no connection', async () => {
  expect.hasAssertions();
  const dataSource = createDataSource();
  try {
    await dataSource.updateData({});
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe('connection not established');
  }
});

test('auto-update fails due to not an object', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect();
  try {
    dataSource.updateData(2 as any as Data);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe('data is not an object');
  }
  await dataSource.disconnect();
});

test('auto-update remove simple succeeds', async () => {
  const dataSource = await createAndConnect({ a: 1 });
  const promise = createConsolePromise();
  dataSource.updateData({});
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [['a']] } }));
  await dataSource.disconnect();
});

test('auto-update remove complex succeeds', async () => {
  const dataSource = await createAndConnect({ foo: { bar: [[{ baz: 1 }, { baz: 2 }]] } });
  const promise = createConsolePromise();
  dataSource.updateData({ foo: { bar: [[{ baz: 1 }, {}]] } });
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [['foo.bar[0][1].baz']] } }));
  await dataSource.disconnect();
});

test('auto-update different removes succeed', async () => {
  const path1 = 'a';
  const path2 = 'b';
  const dataSource = await createAndConnect({ [path1]: 1, [path2]: 2 });
  const promise = createConsolePromise();
  dataSource.updateData({});
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path2], [path1]] } }));
  await dataSource.disconnect();
});

test('auto-update remove and update succeeds', async () => {
  const dataSource = await createAndConnect({ a: 1, b: 2 });
  const promise = createConsolePromise();
  const path1 = 'a';
  const path2 = 'b';
  const value = 3;
  dataSource.updateData({ [path2]: value });
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path2, value], [path1]] } }));
  await dataSource.disconnect();
});

test('auto-update add object property succeeds', async () => {
  const dataSource = await createAndConnect();
  const promise = createConsolePromise();
  const path = 'three';
  const value = 5;
  dataSource.updateData({
    foo: {
      bar: [[
        {
          one: [1, 2]
        },
        {
          two: [3, 4],
          [path]: value
        }
      ]]
    }
  });
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [['foo.bar[0][1].' + path, value]] } }));
  await dataSource.disconnect();
});

test('auto-update array succeeds', async () => {
  const dataSource = await createAndConnect({ foo: { bar: [[{ baz: 1 }, { baz: 2 }]] } });
  const promise = createConsolePromise();
  const path = 'bar';
  const value = 3;
  dataSource.updateData({ foo: { [path]: [value] } });
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[`foo.${path}[0]`, value]] } }));
  await dataSource.disconnect();
});

test('auto-update reorder array succeeds', async () => {
  const path = 'foo';
  const array = [1, 2, 3, 4, 5];
  const dataSource = await createAndConnect({ [path]: array });
  const promise = createConsolePromise();
  array.push(array.shift()!);
  dataSource.updateData({ [path]: array });
  const delta = array.map((value, index) => [`${path}[${index}]`, value]);
  expect(await promise).toBe(JSON.stringify({ debug: { delta } }));
  await dataSource.disconnect();
});

test('auto-update shrink array succeeds', async () => {
  const path = 'foo';
  const array = [1, 2, 3, 4, 5];
  const data = { [path]: array };
  const dataSource = await createAndConnect(data);
  const promise = createConsolePromise();
  array.splice(1, 1);
  dataSource.updateData(data);
  expect(await promise).toBe(JSON.stringify({ debug: { refresh: { data } } }));
  await dataSource.disconnect();
});

test('double-auto-update succeeds', async () => {
  const dataSource = await createAndConnect({});
  const promise = createConsolePromise();
  const path = 'a';
  dataSource.updateData({ [path]: 1 });
  const value = 2;
  dataSource.updateData({ [path]: value });
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path, value]] } }));
  await dataSource.disconnect();
});

test('different auto-updates succeed', async () => {
  const dataSource = await createAndConnect({});
  const promise = createConsolePromise();
  const path1 = 'a';
  const value1 = 1;
  dataSource.updateData({ [path1]: value1 });
  const path2 = 'b';
  const value2 = 2;
  dataSource.updateData({ [path1]: value1, [path2]: value2 });
  expect(await promise).toBe(JSON.stringify({ debug: { delta: [[path1, value1], [path2, value2]] } }));
  await dataSource.disconnect();
});

test('large array change auto-update succeeds', async () => {
  const initialData = JSON.parse(await readFile('__tests__/large-array-0.json'));
  const newData = JSON.parse(await readFile('__tests__/large-array-1.json'));
  const dataSource = await createAndConnect(initialData);
  const promise = createConsolePromise();
  dataSource.updateData(newData);
  const actual = await promise;
  expect(JSON.parse(actual).debug.delta.length).toBe(208);
  await dataSource.disconnect();
});

test('large array change auto-update succeeds as refresh', async () => {
  const initialData = JSON.parse(await readFile('__tests__/large-array-0.json'));
  initialData.allPlayers.push(...initialData.allPlayers);
  initialData.allPlayers.push(...initialData.allPlayers);
  const newData = JSON.parse(await readFile('__tests__/large-array-1.json'));
  newData.allPlayers.push(...newData.allPlayers);
  newData.allPlayers.push(...newData.allPlayers);
  const dataSource = await createAndConnect(initialData);
  const promise = createConsolePromise();
  dataSource.updateData(newData);
  expect(await promise).toBe('delta is too large');
  expect(await createConsolePromise()).toBe(JSON.stringify({ debug: { refresh: { data: newData } } }));
  await dataSource.disconnect();
});

test('auto-update fails due to large message', async () => {
  const dataSource = await createAndConnect({});
  const path = 'a';
  const value = Array(4444).fill('abc').join('def');
  const newData = { [path]: value };
  try {
    dataSource.updateData(newData);
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe(`delta is too large for "${path}"`);
  }
  await dataSource.disconnect();
});

test('auto-update fails due to invalid _metadata', async () => {
  expect.hasAssertions();
  const dataSource = await createAndConnect({ a: 1 });
  try {
    dataSource.updateData({ _metadata: 1 });
    consoleError('unexpected success');
  } catch (ex) {
    expect(ex.message).toBe('_metadata field is not an object');
  }
  await dataSource.disconnect();
});
