import * as fs from 'fs';
import createDataSource, { Configuration, IDataSource, Data, TokenRefreshFn } from '../lib/index';

const trueConsoleError: Function = console.error;
let redirectedConsoleError: Function = console.error;

export function replaceConsoleError(): Function {
  console.error = mockConsoleError;
  return trueConsoleError;
}

function mockConsoleError(message?: any, ...optional: any[]) {
  redirectedConsoleError(message, ...optional);
}

export function createTokenExpiredPromise(configuration: Configuration, timeout?: number): Promise<void> {
  return new Promise((resolve, reject) => {
    const timerId = setTimeout(() => {
      clearHandlers();
      reject('timeout');
    }, timeout || 2222);
    configuration.onTokenExpired = (_fn: TokenRefreshFn) => {
      clearHandlers();
      resolve();
      return false;
    };

    function clearHandlers() {
      clearTimeout(timerId);
      configuration.onTokenExpired = (_fn: TokenRefreshFn) => {
        trueConsoleError('unexpected onTokenExpired invocation');
        return false;
      };
    }
  });
}

export function createConsolePromise(timeout?: number): Promise<string> {
  return new Promise<string>((resolve, _reject) => {
    const error = redirectedConsoleError;
    const timerId = setTimeout(() => {
      clearHandlers();
      resolve('timeout');
    }, timeout || 2222);
    redirectedConsoleError = (name: string, data: string, value: any) => {
      clearHandlers();
      resolve(name === 'DataSource.onMessage' && data === 'unexpected exception' ? value.message.replace('unexpected response from server:  ', '') : data);
    };

    function clearHandlers() {
      clearTimeout(timerId);
      redirectedConsoleError = error;
    }
  });
}

export function createConfiguration(initialData?: Data): Configuration {
  return {
    broadcasterIds: ['1'],
    environment: 'dev',
    gameId: 'test',
    initialData: initialData || {
      foo: {
        bar: [[
          {
            one: [1, 2],
          },
          {
            two: [3, 4],
          },
        ]],
      },
    },
    token: 'app',
    onTokenExpired: (_fn: TokenRefreshFn) => {
      trueConsoleError('unexpected token refresh call-back');
      return false;
    },
  };
}

export async function createAndConnect(initialData?: Data): Promise<IDataSource> {
  const connectionPromise = createConsolePromise();
  const dataSource = createDataSource();
  const configuration = createConfiguration(initialData);
  await dataSource.connect(configuration);
  await connectionPromise;
  return dataSource;
}

export async function readFile(filePath: string): Promise<string> {
  return new Promise<string>((resolve, reject) => {
    fs.readFile(filePath, 'utf8', callback);

    function callback(error: any, data: string) {
      if (error) {
        reject(error);
      } else {
        resolve(data);
      }
    }
  });
}