'use strict';

import { diff } from 'deep-diff';
import isObject from 'isobject';
import WebSocket from 'isomorphic-ws';
import { constructPath, validateData, createConnectMessage, getSegment, sleep, generateSessionId, validatePath } from './utilities';
import { IsTooLarge, MessageType, QueueMessage, createMessageQueue } from './queue';

export type TokenRefreshFn = (token: string) => void;
export type TokenExpiredFn = (fn: TokenRefreshFn) => boolean;

export type Configuration = {
  token: string;
  broadcasterIds?: string[];
  gameId: string;
  environment: string;
  onTokenExpired: TokenExpiredFn;
  initialData: Data;
  sessionId?: string;
  timeoutMs?: number;
  debugFn?: boolean | Function;
};

export type Data = { [key: string]: any };

export interface IDataSource {
  connect: (configuration: Configuration) => Promise<string>;
  disconnect: () => Promise<void>;
  removeField: (path: string) => void;
  updateField: (path: string, value: any) => void;
  updateData: (data: Data) => void;
  appendToArrayField: (path: string, values: any[]) => void;
}

const maximumConnectMessageSize = 100000;
const sendDelay: number = ((s: string): number => {
  const n = parseInt(s, 10);
  return Number.isFinite(n) && n > 0 ? n : 1000;
})(process.env['TWITCH_TEST_ENHANCED_EXPERIENCES_SEND_DELAY'] || '');
const url: string = process.env['TWITCH_TEST_ENHANCED_EXPERIENCES_URL'] || 'wss://metadata.twitch.tv/api/ingest';

export default function createDataSource(): IDataSource {
  let currentData: Data | undefined;
  let webSocket: WebSocket;
  let sessionId: string;
  let token: string;
  let createConnectMessageFn: (data: Data) => Data;
  let connectFn: (data: Data) => Error | undefined;
  let openFn: () => Promise<void>;
  let onTokenExpired: TokenExpiredFn;
  let senderTask: any;
  let lastSendTime: number;
  let debugFn: Function = console.error;
  const queue = createMessageQueue();

  return {
    connect,
    disconnect,
    removeField,
    updateField,
    updateData,
    appendToArrayField,
  };

  async function connect(configuration: Configuration): Promise<string> {
    if (currentData) {
      throw new Error('already connected');
    }

    // Validate the configuration.
    if (typeof (configuration.onTokenExpired) !== 'function') {
      throw new Error('on_token_expired is not callable');
    }
    const data = validateData(configuration.initialData);
    onTokenExpired = configuration.onTokenExpired;
    sessionId = configuration.sessionId || generateSessionId();
    token = configuration.token.valueOf().toString();

    // Compose the connect function.
    const broadcasterIds = configuration.broadcasterIds && configuration.broadcasterIds.length ?
      configuration.broadcasterIds.map((value) => value.valueOf().toString()) :
      undefined;
    const gameId = configuration.gameId.valueOf().toString();
    const environment = configuration.environment.valueOf().toString();
    const isDebug = Boolean(configuration.debugFn);
    if (typeof configuration.debugFn === 'function') {
      debugFn = configuration.debugFn;
    }
    createConnectMessageFn = (data: Data) => createConnectMessage(sessionId, token, broadcasterIds, gameId, environment, isDebug, data);
    const connectMessageSize = JSON.stringify(createConnectMessageFn(data)).length;
    if (connectMessageSize > maximumConnectMessageSize) {
      throw new Error('initial data object is too large');
    }
    connectFn = (data: Data): Error | undefined => {
      try {
        // Send the "Connect" message.
        send(createConnectMessageFn(data));
      } catch (ex) {
        return ex;
      }
    };

    // Compose the open function.
    const timeoutMs = (configuration.timeoutMs || 9999).valueOf();
    openFn = async (): Promise<void> => {
      // Create the socket and await a connection.
      webSocket = await new Promise<WebSocket>((resolve, reject) => {
        const webSocket = new WebSocket(url);
        webSocket.addEventListener('error', onError);
        webSocket.addEventListener('open', onOpen);
        const timerId = setTimeout(onTimeout, timeoutMs);
        function onError(event: WebSocket.ErrorEvent) {
          clearHandlers();
          reject(event.error);
        }
        function onOpen(_event: WebSocket.OpenEvent) {
          clearHandlers();
          resolve(webSocket);
        }
        function onTimeout() {
          clearHandlers();
          reject(new Error('timeout expired'));
        }
        function clearHandlers() {
          webSocket.removeEventListener('error', onError);
          webSocket.removeEventListener('open', onOpen);
          clearTimeout(timerId);
        }
      });
      webSocket.addEventListener('close', onClose);
      webSocket.addEventListener('message', onMessage);
    };

    // Open the WebSocket.
    await openFn();

    // Enqueue a "Reauthorize" message without clearing the token.
    queue.enqueue(QueueMessage.MakeReauthorize(false));

    // Set the current data.
    currentData = data;

    // Invoke sendMessage to send the intial connect message.  The "Reauthorize"
    // message handler will start the sender task.
    await sendMessage();
    return sessionId;
  }

  async function disconnect(): Promise<void> {
    clearTimeout(senderTask);
    senderTask = undefined;
    if (queue.some()) {
      queue.replaceWith(QueueMessage.Refresh);
      const nextSendTime = lastSendTime + sendDelay;
      const finalDelay = nextSendTime - Date.now();
      if (finalDelay > 0) {
        await sleep(finalDelay);
      }
      await sendMessage();
    }
    closeWebSocket();
    currentData = undefined;
  }

  function removeField(path: string) {
    // Validate the connection and path.
    validateConnection();
    path = validatePath(path);
    if (path.endsWith(']')) {
      throw new Error(`"${path}" does not specify a field`);
    }
    const segment = getSegment(currentData!, path);
    if (!segment || typeof segment.parent[segment.field] === 'undefined') {
      debugFn('DataSource.remove_field', 'warning:  ignoring removal of an unknown field', path);
      return;
    }

    // Enqueue the "Remove Field" message.
    queue.enqueue(new QueueMessage(MessageType.Remove, path));

    // Update the current data.
    const parent = segment.parent as Data;
    const field = segment.field as string;
    delete parent[field];
  }

  function updateField(path: string, value: any) {
    // Validate the connection, path, and value.
    validateConnection();
    path = validatePath(path);
    value = JSON.parse(JSON.stringify(value));

    // If updating the _metadata field, ensure it is valid.
    if (path === '_metadata') {
      validateData({ [path]: value });
    }

    // Verify the path results in a value.
    const newData = JSON.parse(JSON.stringify(currentData!));
    const segment = getSegment(newData, path);
    if (!segment) {
      throw new Error(`"${path}" does not specify a known field`);
    } else if (Array.isArray(segment.parent) && segment.field >= segment.parent.length) {
      throw new Error(`"${path}" is out of bounds`);
    }
    if (segment.parent[segment.field] === value) {
      // Skip update if the value did not change.
      return;
    }
    segment.parent[segment.field] = value;
    checkConnectMessageSize(newData);

    // Enqueue the "Update" message.
    queue.enqueue(new QueueMessage(MessageType.Update, path, value));

    // Update the current data.
    currentData = newData;
  }

  function updateData(data: Data) {
    // Validate the connection and data.
    validateConnection();
    data = validateData(data);

    // Produce deltas for this update only if all current messages in the queue
    // are delta messages.  Otherwise, there is a message in the queue that will
    // invoke the equivalent of a "Refresh" message.
    if (queue.every((m) => m.isDelta())) {
      // Get differences.
      const differences = diff(currentData, data);
      if (differences) {
        // Enqueue appropriate modification messages.
        const localQueue: QueueMessage[] = [];
        const result = differences.every((difference) => {
          const path = constructPath(difference.path || []);
          switch (difference.kind) {
            case 'A':
              if (difference.item.kind === 'N') {
                localQueue.push(new QueueMessage(MessageType.Append, path, [difference.item.rhs]));
              } else {
                return false;
              }
              break;
            case 'D':
              localQueue.push(new QueueMessage(MessageType.Remove, path));
              break;
            case 'E':
            case 'N':
              localQueue.push(new QueueMessage(MessageType.Update, path, difference.rhs));
              break;
            default:
              debugFn('DataSource.updateData', 'unexpected difference', difference);
              return false;
          }
          return true;
        });
        if (result) {
          // Reverse the local queue so array additions are in the correct order.
          queue.enqueue(...localQueue.reverse());

          // Enqueue a "Refresh" message if this update and whatever is in the queue
          // cannot be sent as a single delta message.
          const deltas = queue.map((m) => m.asDelta());
          if (IsTooLarge(deltas)) {
            debugFn('DataSource.sendMessage', 'delta is too large', queue.peek());
            queue.replaceWith(new QueueMessage(MessageType.Refresh));
          }
        } else {
          // The changes cannot be represented with updates.  Enqueue a "Refresh" message.
          queue.enqueue(QueueMessage.Refresh);
        }
      }
    }

    // Update the current data.
    currentData = data;
  }

  function appendToArrayField(path: string, values: any[]) {
    // Validate the connection, path, and values.
    validateConnection();
    path = validatePath(path);
    values = JSON.parse(JSON.stringify(values));
    if (!Array.isArray(values)) {
      throw new Error('values is not an array');
    }
    const newData = JSON.parse(JSON.stringify(currentData!));
    const segment = getSegment(newData, path);
    if (!segment || typeof segment.parent[segment.field] === 'undefined') {
      throw new Error(`"${path}" does not specify a known field`);
    }
    const array = segment.parent[segment.field];
    if (!Array.isArray(array)) {
      throw new Error(`"${path}" does not specify an array field`);
    } else if (values.length === 0) {
      debugFn('DataSource.appendToArrayField', 'warning:  array is empty; ignoring', { path, values });
      return;
    }
    array.push(...values);
    checkConnectMessageSize(newData);

    // Enqueue the "Append" message.
    queue.enqueue(new QueueMessage(MessageType.Append, path, values));

    // Update the current data.
    currentData = newData;
  }

  async function acquireToken(): Promise<string> {
    return await new Promise<string>((resolve, _reject) => {
      try {
        if (!onTokenExpired(resolve)) {
          // The client wants to shut down.
          resolve('');
        }
      } catch (ex) {
        // There is likely a logic error in the client.  Shut down the connection.
        debugFn('DataSource.acquireToken', 'onTokenExpired threw an exception', ex);
        resolve('');
      }
    });
  }

  async function sendMessage(): Promise<void> {
    try {
      senderTask = undefined;
      const uniqueMessage = queue.findUniqueMessage();
      if (uniqueMessage) {
        // There is a unique message.  Clear the queue and process the message.
        queue.clear();
        switch (uniqueMessage.type) {
          case MessageType.Reauthorize:
            if (uniqueMessage.value) {
              token = '';
            }
            await reauthorize();
            break;
          case MessageType.Reconnect:
            // Await the requested delay.
            const reconnectDelay = uniqueMessage.value - Date.now();
            if (reconnectDelay > 0) {
              await sleep(reconnectDelay);
            }

            if (currentData) {
              // Open a new connection.
              await openFn();

              // Perform the "Reauthorize" action above.
              if (currentData) {
                await reauthorize();
              }
            }
            break;
          case MessageType.Refresh:
            send({ refresh: { data: currentData } });
            break;
          default:
            throw new Error(`unexpected message type ${uniqueMessage.type}`);
        }
      } else {
        // Trim the queue based on repeated modifications.
        const trimmedQueue: QueueMessage[] = [];
        while (queue.some()) {
          const message = queue.dequeue()!;
          switch (message.type) {
            case MessageType.Append:
              const currentMessage = trimmedQueue.find((m) => m.type === message.type && m.path === message.path);
              if (currentMessage) {
                currentMessage.value.push(...message.value);
              } else {
                trimmedQueue.push(message);
              }
              break;
            case MessageType.Remove:
            case MessageType.Update:
              trimmedQueue.splice(0, trimmedQueue.length, ...trimmedQueue.filter((m) => m.path !== message.path));
              trimmedQueue.push(message);
              break;
            default:
              throw new Error(`unexpected message type ${message.type}`);
          }
        }
        queue.replaceWith(...trimmedQueue);

        // Take messages off of the queue until reaching the maximum payload size.
        const deltas: any[] = [];
        while (queue.some()) {
          const message = queue.peek();
          const delta = message.asDelta();
          if (IsTooLarge([...deltas, delta])) {
            break;
          }
          deltas.push(delta);
          queue.dequeue();
        }
        if (deltas.length) {
          send({ delta: deltas });
        }
      }
    } catch (ex) {
      debugFn('DataSource.sendMessage', 'unexpected exception', ex);
    } finally {
      // Set another time-out to send the next message.
      if (currentData) {
        senderTask = setTimeout(sendMessage, sendDelay);
      }
    }
  }

  async function reauthorize(): Promise<boolean> {
    if (!token) {
      token = await acquireToken();
    }
    if (token) {
      const error = connectFn(currentData!);
      if (!error) {
        return true;
      }
      debugFn('DataSource.reauthorize', 'connection error', error);
    }
    await disconnect();
    return false;
  }

  function validateConnection() {
    // Ensure connect has already been invoked.
    if (!currentData) {
      throw new Error('connection not established');
    }
  }

  function send(message: Data) {
    webSocket.send(JSON.stringify(message));
    lastSendTime = Date.now();
  }

  function checkConnectMessageSize(data: Data) {
    const s = JSON.stringify(createConnectMessageFn(data));
    if (s.length > maximumConnectMessageSize) {
      throw new Error('data object is too large');
    }
  }

  function closeWebSocket() {
    if (webSocket) {
      webSocket.removeEventListener('close', onClose);
      webSocket.removeEventListener('message', onMessage);
      webSocket.close();
      webSocket = undefined as unknown as WebSocket;
    }
  }

  function onClose(_event: WebSocket.CloseEvent) {
    // Clear the handlers and enqueue a "Reconnect" message.
    webSocket.removeEventListener('close', onClose);
    webSocket.removeEventListener('message', onMessage);
    queue.enqueue(QueueMessage.MakeReconnect());
  }

  function onMessage(event: WebSocket.MessageEvent) {
    try {
      const data = event.data.toString();
      const response = JSON.parse(data);
      const keys = Object.keys(response);
      if (keys.length === 1) {
        switch (keys[0]) {
          case 'connected':
            if (response['connected']) {
              return;
            }
            break;
          case 'error':
            const error = response['error'];
            if (isObject(error)) {
              const code = error['code'];
              if (code === 'invalid_connect_token') {
                // The server has rejected the authorization token.
                queue.enqueue(QueueMessage.MakeReauthorize(true));
                return;
              } else if (code === 'connection_not_authed') {
                // Ignore this since it means we're in the middle of getting connected.
                return;
              } else if (code === 'waiting_on_refresh_message') {
                // Ignore this since it means we're in the middle of refreshing.
                return;
              }
            }
            break;
          case 'reconnect':
            const reconnectDelay = response['reconnect'];
            if (typeof reconnectDelay === 'number') {
              // The server requested a reconnection.  Close the socket.  The reconnection
              // process will open a new one.
              closeWebSocket();
              queue.enqueue(QueueMessage.MakeReconnect(reconnectDelay));
              return;
            }
            break;
        }
      }
      throw new Error(`unexpected response from server:  ${data}`);
    } catch (ex) {
      debugFn('DataSource.onMessage', 'unexpected exception', ex);
    }
  }
}
