'use strict';

export enum MessageType {
  Append,
  Reauthorize,
  Reconnect,
  Refresh,
  Remove,
  Update,
}

const maximumDeltaSize = 20000;

export function IsTooLarge(deltas: any[]) {
  return JSON.stringify({ delta: deltas }).length > maximumDeltaSize;
}

export class QueueMessage {
  public type: MessageType;
  public path?: string;
  public value?: any;

  public static MakeReauthorize(clearingToken: boolean) {
    return new QueueMessage(MessageType.Reauthorize, undefined, clearingToken);
  }
  public static MakeReconnect(reconnectDelay?: number) {
    const reconnectTime = (reconnectDelay || 0) + Date.now();
    return new QueueMessage(MessageType.Reconnect, undefined, reconnectTime);
  }
  public static Refresh: QueueMessage = new QueueMessage(MessageType.Refresh);

  constructor(type: MessageType, path?: string, value?: any) {
    this.type = type;
    this.path = path;
    this.value = value;
    if (this.isDelta() && IsTooLarge([this.asDelta()])) {
      throw new Error(`delta is too large for "${path || type}"`);
    }
  }

  public asDelta(): any[] {
    switch (this.type) {
      case MessageType.Append:
        return [this.path, 'a', this.value];
      case MessageType.Remove:
        return [this.path];
      case MessageType.Update:
        return [this.path, this.value];
      default:
        throw new Error(`unexpected message type ${this.type}`);
    }
  }

  public isDelta(): boolean {
    return [MessageType.Append, MessageType.Remove, MessageType.Update].some((m) => m === this.type);
  }
}

export interface IMessageQueue {
  clear: () => void;
  dequeue: () => QueueMessage | undefined;
  findUniqueMessage: () => QueueMessage | undefined;
  peek: () => QueueMessage;
  enqueue: (...message: QueueMessage[]) => void;
  replaceWith: (...message: QueueMessage[]) => void;
  some: () => boolean;
  every: (fn: (value: QueueMessage) => boolean) => boolean;
  map: (fn: (value: QueueMessage) => any) => any[];
}

// These are in priority order.
const uniqueMessageTypes: MessageType[] = [MessageType.Reconnect, MessageType.Reauthorize, MessageType.Refresh];

export function createMessageQueue(): IMessageQueue {
  const queue: QueueMessage[] = [];

  return {
    clear,
    dequeue,
    findUniqueMessage,
    peek,
    enqueue,
    replaceWith,
    some,
    every,
    map,
  };

  function clear() {
    queue.splice(0, queue.length);
  }

  function dequeue() {
    return queue.shift();
  }

  function findUniqueMessage() {
    for (const uniqueMessageType of uniqueMessageTypes) {
      const uniqueMessage = queue.find((message) => message.type === uniqueMessageType);
      if (uniqueMessage) {
        return uniqueMessage;
      }
    }
  }

  function peek() {
    return queue[0];
  }

  function enqueue(...message: QueueMessage[]) {
    queue.push(...message);
  }

  function replaceWith(...message: QueueMessage[]) {
    queue.splice(0, queue.length, ...message);
  }

  function some() {
    return queue.length > 0;
  }

  function every(fn: (value: QueueMessage) => boolean) {
    return queue.every(fn);
  }

  function map(fn: (value: QueueMessage) => any) {
    return queue.map(fn);
  }
}
