'use strict';

import isObject from 'isobject';
import { Data } from '.';

export function constructPath(path: any[]): string {
  return path.reduce((path, segment) => {
    return path + (typeof segment === 'number' ? `[${segment}]` : path ? '.' + segment : segment);
  }, '');
}

export function createConnectMessage(sessionId: string, token: string, broadcasterIds: string[] | undefined, gameId: string, environment: string, isDebug: boolean, data: Data): Data {
  const message: Data = {
    connect: {
      session_id: sessionId,
      token,
      game_id: gameId,
      env: environment,
      data,
    },
  };
  if (broadcasterIds) {
    message.connect.broadcaster_ids = broadcasterIds;
  }
  if (isDebug) {
    message.connect.debug = true;
  }
  return message;
}

export function getSegment(data: Data, path: string): { parent: Data, field: string | number } | undefined {
  const parts = path.split(/[.[]/);
  const field = parts.pop();
  if (field) {
    const parent = parts.reduce(checkSegment, data);
    if (parent) {
      if (field.endsWith(']')) {
        if (Array.isArray(parent)) {
          const index = parseInt(field, 10);
          return { parent, field: index };
        }
      } else if (isObject(parent)) {
        return { parent, field };
      }
    }
  }

  function checkSegment(data: Data | undefined, segment: string): Data | undefined {
    if (data) {
      if (segment.endsWith(']')) {
        if (Array.isArray(data)) {
          const index = parseInt(segment, 10);
          return data[index];
        }
      } else if (isObject(data)) {
        return data[segment];
      }
    }
  }
}

export async function sleep(milliseconds: number) {
  await new Promise((resolve, _reject) => setTimeout(resolve, milliseconds));
}

export function validateData(data: Data): Data {
  // Validate the data.
  if (!isObject(data)) {
    throw new Error('data is not an object');
  }

  // If there is a "_metadata" field, ensure it is an object.
  const metadata = data['_metadata'];
  if (metadata !== undefined && !isObject(metadata)) {
    throw new Error('_metadata field is not an object');
  }

  return JSON.parse(JSON.stringify(data));
}

const pathRx = /^\w+(\.\w+|\[[0-9]\])*$/;

export function validatePath(path: string): string {
  path = path.valueOf();
  if (!path) {
    throw new Error('path is empty');
  } else if (typeof path !== 'string') {
    throw new Error('path is not a string');
  } else if (!pathRx.test(path)) {
    throw new Error(`"${path}" is not a valid field specifier`);
  }
  return path;
}

const idCharacters: string = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz';

export function generateSessionId() {
  return Array(32).fill(0).map(_ => idCharacters.charAt(Math.floor(idCharacters.length * Math.random()))).join('');
}
