#include "pch.h"
#include "Utilities.h"

using namespace std::literals;
using namespace Microsoft::VisualStudio::CppUnitTestFramework;
using namespace Twitch;

namespace WindowsLibTest {
	TEST_CLASS(AppendTest) {
public:
	TEST_METHOD(Fails_ConnectMessageIsTooLarge) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration(_T("{\"a\":[\"") + tstring(88'888, _T('a')) + _T("\"]}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.AppendToArrayFieldWithJson(_T("a"s), "[\"" + std::string(11'111, 'a') + "\"]");
		});
	}

	TEST_METHOD(Fails_DataSourceIsNotConnected) {
		DataSource dataSource;
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.AppendToArrayFieldWithJson(_T("a"s), "[1]"s);
		});
	}

	TEST_METHOD(Fails_DeltaMessageIsTooLarge) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.AppendToArrayFieldWithJson(_T("a"s), "[\"" + std::string(22'222, 'a') + "\"]");
		});
	}

	TEST_METHOD(Fails_MetadataFieldIsNotAnArray) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration(_T("{\"_metadata\":{\"a\":1}}"s));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.AppendToArrayFieldWithJson(_T("_metadata"s), "[1]"s);
		});
	}

	TEST_METHOD(Fails_PathIsEmpty) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.AppendToArrayFieldWithJson(_T(""s), "[1]"s);
		});
	}

	TEST_METHOD(Fails_PathIsMalformed) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.AppendToArrayFieldWithJson(_T("a[0"s), "[1]"s);
		});
	}

	TEST_METHOD(Fails_TargetFieldIsMissing) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.AppendToArrayFieldWithJson(_T("b"s), "[1]"s);
		});
	}

	TEST_METHOD(Fails_TargetFieldIsNotAnArray) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration(_T("{\"a\":1}"s));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.AppendToArrayFieldWithJson(_T("a"s), "[1]"s);
		});
	}

	TEST_METHOD(Fails_ValueIsNotAnArray) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.AppendToArrayFieldWithJson(_T("a"s), "1"s);
		});
	}

	TEST_METHOD(Fails_ValueIsNotJson) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.AppendToArrayFieldWithJson(_T("a"s), "$"s);
		});
	}

	TEST_METHOD(Succeeds_TargetArrayIsChanged) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("echo"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto const path = _T("a"s);
		auto const value = "[1]"s;
		dataSource.AppendToArrayFieldWithJson(path, value);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"debug\":{\"delta\":[[\"") +
			path + _T("\",\"a\",") + ToTstring(value) + _T("]]}}\n");
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Succeeds_TargetArrayIsChangedTwice) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("echo"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto const path = _T("a"s);
		auto const value1 = "[2]"s;
		auto const value2 = "[3]"s;
		dataSource.AppendToArrayFieldWithJson(path, value1);
		dataSource.AppendToArrayFieldWithJson(path, value2);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"debug\":{\"delta\":[[\"") +
			path + _T("\",\"a\",") + ToTstring(value1.substr(0, value1.size() - 1)) + _T(',') +
			ToTstring(value2.substr(1)) + _T("]]}}\n");
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Succeeds_ValueIsEmptyArray) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("echo"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		dataSource.AppendToArrayFieldWithJson(_T("a"s), "[]"s);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSource::AppendToArrayFieldWithJson] warning:  array is empty; ignoring\n"s);
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

private:
	DataSource::Configuration MakeConfiguration(tstring const& initialData = _T("{\"a\":[1]}"s)) {
		DataSource::Configuration configuration;
		configuration.environment = _T("dev"s);
		configuration.gameId = _T("test"s);
		configuration.initialData = initialData;
		configuration.isDebug = true;
		configuration.token = _T("token"s);
		configuration.onTokenExpired = [](DataSource::TokenRefreshFn) {
			Assert::Fail(_T("unexpected token refresh call-back"));
			return false;
		};
		return configuration;
	}
	};
}
