#include "pch.h"
#include "Utilities.h"

using namespace std::literals;
using namespace Microsoft::VisualStudio::CppUnitTestFramework;
using namespace Twitch;

namespace WindowsLibTest {
	TEST_CLASS(ConnectTest) {
public:
	TEST_METHOD(Fails_CallbackNotProvided) {
		auto future = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration(_T("bad-token"s));
		configuration.onTokenExpired = DataSource::TokenExpiredFn();
		dataSource.Connect(configuration);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::AcquireToken] warning:  onTokenExpired not provided\n"s);
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Fails_ClientNotWhitelisted) {
		auto future = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration(_T("error-connect_client_not_whitelisted"s));
		dataSource.Connect(configuration);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected error from server:  connect_client_not_whitelisted\n"s);
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Fails_ConnectMessageTooLarge) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.initialData = _T("{\"a\":\"") + tstring(111'111, _T('a')) + _T("\"}");
		Assert::ExpectException<TwitchException>([&dataSource, &configuration] {
			dataSource.Connect(configuration);
		});
	}

	TEST_METHOD(Fails_GoodTokenNotProvided) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration(_T("bad-token"s));
		std::promise<void> promise;
		configuration.onTokenExpired = [&promise](auto) {
			promise.set_value();
			return false;
		};
		dataSource.Connect(configuration);
		Assert::IsTrue(promise.get_future().wait_for(1110ms) != std::future_status::timeout);
	}

	TEST_METHOD(Fails_InitialDataNotObject) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.initialData = _T("1"s);
		Assert::ExpectException<TwitchException>([&dataSource, &configuration] {
			dataSource.Connect(configuration);
		});
	}

	TEST_METHOD(Fails_InitialDataIsNotJson) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.initialData = _T("$"s);
		Assert::ExpectException<TwitchException>([&dataSource, &configuration] {
			dataSource.Connect(configuration);
		});
	}

	TEST_METHOD(Fails_InvalidMetadata) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.initialData = _T("{\"_metadata\":1}"s);
		Assert::ExpectException<TwitchException>([&dataSource, &configuration] {
			dataSource.Connect(configuration);
		});
	}

	TEST_METHOD(Fails_InvalidResponse) {
		auto future = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("unexpected-payload-$"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] invalid JSON from server:  expected value, got '$' (36)\n"s);
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Fails_MissingToken) {
		auto future = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration(_T(""s));
		configuration.onTokenExpired = DataSource::TokenExpiredFn();
		dataSource.Connect(configuration);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::AcquireToken] warning:  onTokenExpired not provided\n"s);
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Fails_Twice) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource, &configuration] {
			dataSource.Connect(configuration);
		});
	}

	TEST_METHOD(Fails_UnexpectedResponse) {
		auto future = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("unexpected-payload-1"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  1\n"s);
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(ConnectAbortSucceeds) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		auto const token = _T("abort"s);
		configuration.token = token;
		dataSource.Connect(configuration);
		ExpectConnected();
		auto failedConnectionFuture = CreateErrorPromise();
		AwaitErrorFuture(failedConnectionFuture);
		auto const expected = _T("[WebSocketImpl::ReceiveMessages] WinHttpWebSocketReceive error 12030\n"s);
		auto const actual = failedConnectionFuture.get();
		Assert::AreEqual(expected, actual);
		ExpectConnected();
	}

	TEST_METHOD(ConnectAppTokenSucceeds) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration(_T("app"s));
		configuration.broadcasterIds = { _T("1"s) };
		auto const sessionId = dataSource.Connect(configuration);
		ExpectConnected();
	}

	TEST_METHOD(Fails_EmptyBroadcasters) {
		auto future = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration(_T("app"s));
		dataSource.Connect(configuration);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected error from server:  connect_invalid_values:  \"broadcaster_ids\"\n"s);
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(ConnectUserTokenSucceeds) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration(_T("user"s));
		auto const sessionId = dataSource.Connect(configuration);
		ExpectConnected();
	}

	TEST_METHOD(Fails_Broadcasters) {
		auto future = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration(_T("user"s));
		configuration.broadcasterIds = { _T("1"s) };
		dataSource.Connect(configuration);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected error from server:  connect_invalid_values:  \"broadcaster_ids\"\n"s);
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Succeeds_Delayed) {
		auto future = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration(_T("bad-token"s));
		auto const* s = _T("token");
		std::future<void> task;
		configuration.onTokenExpired = [s, &task](DataSource::TokenRefreshFn fn) {
			task = std::async(std::launch::async, [s, fn] {
				std::this_thread::sleep_for(11ms);
				fn(s);
				test_OutputDebugString(s);
			});
			return true;
		};
		dataSource.Connect(configuration);
		AwaitErrorFuture(future);
		tstring expected(s);
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Succeeds_Immediate) {
		auto future = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration(_T("bad-token"s));
		auto const* s = _T("token");
		configuration.onTokenExpired = [s](DataSource::TokenRefreshFn fn) {
			fn(s);
			test_OutputDebugString(s);
			return true;
		};
		dataSource.Connect(configuration);
		AwaitErrorFuture(future);
		tstring expected(s);
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Succeeds_Long) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		auto const token = _T("restart-long"s);
		configuration.token = token;
		dataSource.Connect(configuration);
		ExpectConnected();
		auto const expected = _T("[WebSocketImpl::ReceiveMessages] WinHttpWebSocketReceive error 12030\n"s);
		auto future = CreateErrorPromise();
		AwaitErrorFuture(future);
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
		ExpectConnected(4s);
	}

	TEST_METHOD(Succeeds_Short) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		auto const token = _T("restart-short"s);
		configuration.token = token;
		dataSource.Connect(configuration);
		ExpectConnected();
		auto const expected = _T("[WebSocketImpl::ReceiveMessages] WinHttpWebSocketReceive error 12030\n"s);
		auto future = CreateErrorPromise();
		AwaitErrorFuture(future);
		auto const actual_ = future.get();
		Assert::AreEqual(expected, actual_);
		ExpectConnected();
	}

private:
	void ExpectConnected(std::chrono::milliseconds timeout = 1110ms) {
		auto initialConnectionFuture = CreateErrorPromise();
		AwaitErrorFuture(initialConnectionFuture, timeout);
	}

	DataSource::Configuration MakeConfiguration(tstring const& token = _T("token"s)) {
		DataSource::Configuration configuration;
		configuration.environment = _T("dev"s);
		configuration.gameId = _T("516575"s);
		configuration.initialData = DataSource::EmptyData;
		configuration.isDebug = true;
		configuration.onTokenExpired = [](DataSource::TokenRefreshFn) {
			Assert::Fail(_T("unexpected token refresh call-back"));
			return false;
		};
		configuration.token = token;
		return configuration;
	}
	};
}
