#include "pch.h"
#include "Utilities.h"

using namespace std::literals;
using namespace Microsoft::VisualStudio::CppUnitTestFramework;
using namespace Twitch;

namespace WindowsLibTest {
	TEST_CLASS(RemoveTest) {
public:
	TEST_METHOD(Fails_DataSourceIsNotConnected) {
		DataSource dataSource;
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.RemoveField(_T("root.one"s));
		});
	}

	TEST_METHOD(Fails_FieldIsNotFound) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.RemoveField(_T("root.three"s));
		});
	}

	TEST_METHOD(Fails_PathIsEmpty) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.RemoveField(_T(""s));
		});
	}

	TEST_METHOD(Fails_PathIsMalformed) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.RemoveField(_T("root.array[0"s));
		});
	}

	TEST_METHOD(Fails_RemoveIsInvokedTwice) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		auto const path = _T("root.one"s);
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("echo"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		dataSource.RemoveField(path);
		Assert::ExpectException<TwitchException>([&dataSource, &path] {
			dataSource.RemoveField(path);
		});
		auto future = CreateErrorPromise();
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"debug\":{\"delta\":[[\"") +
			path + _T("\"]]}}\n");
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Fails_TargetIsArrayElement) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.RemoveField(_T("root.array[0]"s));
		});
	}

	TEST_METHOD(Succeeds_FieldIsRemoved) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("echo"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto const path = _T("root.one"s);
		dataSource.RemoveField(path);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"debug\":{\"delta\":[[\"") +
			path + _T("\"]]}}\n");
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Succeeds_FieldIsRemovedAfterAppend) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("echo"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto const path = _T("root.array"s);
		dataSource.AppendToArrayFieldWithJson(path, "[0]"s);
		dataSource.RemoveField(path);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"debug\":{\"delta\":[[\"") +
			path + _T("\"]]}}\n");
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Succeeds_FieldIsRemovedAfterUpdate) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("echo"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto const path = _T("a"s);
		dataSource.UpdateFieldWithJson(path, "1"s);
		dataSource.RemoveField(path);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"debug\":{\"delta\":[[\"") +
			path + _T("\"]]}}\n");
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Succeeds_MetadataIsRemoved) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("echo"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto const path = _T("_metadata"s);
		dataSource.RemoveField(path);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"debug\":{\"delta\":[[\"") +
			path + _T("\"]]}}\n");
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Succeeds_OneFieldIsRemovedAfterAnother) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		auto const path1 = _T("root.one"s);
		auto const path2 = _T("root.two"s);
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("echo"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		dataSource.RemoveField(path1);
		dataSource.RemoveField(path2);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"debug\":{\"delta\":[[\"") +
			path1 + _T("\"],[\"") + path2 + _T("\"]]}}\n");
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

private:
	DataSource::Configuration MakeConfiguration() {
		DataSource::Configuration configuration;
		configuration.environment = _T("dev"s);
		configuration.gameId = _T("test"s);
		configuration.initialData = _T("{\"_metadata\":{\"value\":0},\"root\":{\"one\":1,\"two\":2,\"array\":[1,2]}}"s);
		configuration.isDebug = true;
		configuration.token = _T("token"s);
		configuration.onTokenExpired = [](DataSource::TokenRefreshFn) {
			Assert::Fail(_T("unexpected token refresh call-back"));
			return false;
		};
		return configuration;
	}
	};
}
