#include "pch.h"
#include "Utilities.h"

using namespace std::literals;
using namespace Microsoft::VisualStudio::CppUnitTestFramework;
using namespace Twitch;

namespace WindowsLibTest {
	TEST_CLASS(UpdateTest) {
public:
	TEST_METHOD(Fails_ArrayIndexIsOutOfBounds) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateFieldWithJson(_T("root.array[2]"s), "-1"s);
		});
	}

	TEST_METHOD(Fails_ConnectMessageIsTooLarge) {
		DataSource dataSource;
		auto const initialData = _T("{\"a\":\"") + tstring(88'888, _T('a')) + _T("\"}");
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			auto const value = '"' + std::string(22'222, 'b') + '"';
			dataSource.UpdateFieldWithJson(_T("b"s), value);
		});
	}

	TEST_METHOD(Fails_DataSourceIsNotConnected) {
		DataSource dataSource;
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateFieldWithJson(_T("root.one"s), "-1"s);
		});
	}

	TEST_METHOD(Fails_DeltaMessageIsTooLarge) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			auto const value = '"' + std::string(22'222, 'a') + '"';
			dataSource.UpdateFieldWithJson(_T("root.one"s), value);
		});
	}

	TEST_METHOD(Fails_MetadataIsNotAnObject) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateFieldWithJson(_T("_metadata"s), "-1"s);
		});
	}

	TEST_METHOD(Fails_PathIsEmpty) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateFieldWithJson(_T(""s), "-1"s);
		});
	}

	TEST_METHOD(Fails_PathIsMalformed) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateFieldWithJson(_T("root.array[0"s), "-1"s);
		});
	}

	TEST_METHOD(Fails_TargetFieldIsAnArray) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateFieldWithJson(_T("root.array.value"s), "-1"s);
		});
	}

	TEST_METHOD(Fails_TargetFieldIsMissing) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateFieldWithJson(_T("a.b"s), "-1"s);
		});
	}

	TEST_METHOD(Fails_TargetFieldIsNotAnArray) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateFieldWithJson(_T("root.one[0]"s), "-1"s);
		});
	}

	TEST_METHOD(Fails_ValueIsNotJson) {
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateFieldWithJson(_T("root.one"s), "$"s);
		});
	}

	TEST_METHOD(Succeeds_ArrayElementIsChanged) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("echo"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto const path = tstring(_T("root.array[1]"s));
		auto const value = "-1"s;
		dataSource.UpdateFieldWithJson(path, value);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"debug\":{\"delta\":[[\"") +
			path + _T("\",") + ToTstring(value) + _T("]]}}\n");
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Succeeds_FieldIsAdded) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("echo"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto const path = tstring(_T("root.three"s));
		auto const value = "-1"s;
		dataSource.UpdateFieldWithJson(path, value);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"debug\":{\"delta\":[[\"") +
			path + _T("\",") + ToTstring(value) + _T("]]}}\n");
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Succeeds_FieldIsAddedAfterRemove) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		auto const path = _T("root.one"s);
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("echo"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		dataSource.RemoveField(path);
		auto const value = "-1"s;
		dataSource.UpdateFieldWithJson(path, value);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"debug\":{\"delta\":[[\"") +
			path + _T("\",") + ToTstring(value) + _T("]]}}\n");
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Succeeds_FieldIsChanged) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("echo"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto const path = tstring(_T("root.one"s));
		auto const value = "-1"s;
		dataSource.UpdateFieldWithJson(path, value);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"debug\":{\"delta\":[[\"") +
			path + _T("\",") + ToTstring(value) + _T("]]}}\n");
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Succeeds_MetadataIsChanged) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("echo"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto const path = tstring(_T("_metadata"s));
		auto const value = "{\"update\":1}"s;
		dataSource.UpdateFieldWithJson(path, value);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"debug\":{\"delta\":[[\"") +
			path + _T("\",") + ToTstring(value) + _T("]]}}\n");
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Succeeds_OneFieldIsUpdatedAfterAnother) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("echo"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto const path1 = tstring(_T("root.one"s));
		auto const value1 = "-1"s;
		dataSource.UpdateFieldWithJson(path1, value1);
		auto const path2 = tstring(_T("root.two"s));
		auto const value2 = "-1"s;
		dataSource.UpdateFieldWithJson(path2, value2);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"debug\":{\"delta\":[[\"") +
			path1 + _T("\",") + ToTstring(value1) + _T("],[\"") + path2 + _T("\",") + ToTstring(value2) + _T("]]}}\n");
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Succeeds_UpdateIsInvokedTwice) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = MakeConfiguration();
		configuration.token = _T("echo"s);
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto const path = tstring(_T("root.one"s));
		dataSource.UpdateFieldWithJson(path, "0"s);
		auto const value = "-1"s;
		dataSource.UpdateFieldWithJson(path, value);
		AwaitErrorFuture(future);
		auto const expected = _T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"debug\":{\"delta\":[[\"") +
			path + _T("\",") + ToTstring(value) + _T("]]}}\n");
		auto const actual = future.get();
		Assert::AreEqual(expected, actual);
	}

private:
	DataSource::Configuration MakeConfiguration() {
		DataSource::Configuration configuration;
		configuration.environment = _T("dev"s);
		configuration.gameId = _T("test"s);
		configuration.initialData = _T("{\"_metadata\":{\"value\":0},\"root\":{\"one\":1,\"two\":2,\"array\":[1,2]}}"s);
		configuration.isDebug = true;
		configuration.token = _T("token"s);
		configuration.onTokenExpired = [](DataSource::TokenRefreshFn) {
			Assert::Fail(_T("unexpected token refresh call-back"));
			return false;
		};
		return configuration;
	}
	};
}
