#include "pch.h"
#include "Display.h"
#include "View.h"
#include "shader_cb.h"
#include "framework/sample_framework.h"
#include "toolkit/toolkit.h"

using namespace sce;
using namespace sce::Gnmx;

namespace {
#define APP_ROOT "/app0/"
	char const* const vertexShaderFile = APP_ROOT "shader_vv.sb";
	char const* const pixelShaderFile = APP_ROOT "shader_p.sb";
}

class MyDisplayImpl : public MyDisplay {
private:
	std::array<View, 2> m_views;
	Framework::GnmSampleFramework m_framework;
	Gnm::Sampler m_sampler;
	Gnmx::GfxContext* m_gfxc;
	Gnmx::VsShader* m_vsb;
	Gnmx::PsShader* m_psb;
	bool m_verticalSplit = false;

public:
	int initialize() override {
		m_framework.m_config.m_renderText = 0;
		m_framework.m_config.m_asynchronous = 0;
		m_framework.m_config.m_onionMemoryInBytes = 1024 * 1024 * 128;
		m_framework.m_config.m_garlicMemoryInBytes = 1024 * 1024 * 1024;

		m_framework.initialize("PlayerCore Sample",
			"Copyright(C) 2019 Twitch Interactive, Inc.",
			"PlayerCore Sample.",
			0, nullptr);

		ON_DBG("GnmSampleFramework::initialize() done ... m_renderText:%d m_asynchronous:%d\n",
			m_framework.m_config.m_renderText,
			m_framework.m_config.m_asynchronous);

		for(uint32_t bCnt = 0; bCnt < m_framework.m_config.m_buffers; ++bCnt) {
			m_framework.m_buffer[bCnt].m_renderTarget.setDataFormat(Gnm::kDataFormatB8G8R8A8Unorm);
		}

		// Set up gfx context
		const uint32_t kNumRingEntries = 64;
		const uint32_t cueHeapSize = Gnmx::ConstantUpdateEngine::computeHeapSize(kNumRingEntries);

		// one GfxContext (command buffer) object is required for each frame that can simultaneously be in flight.
		m_gfxc = new Gnmx::GfxContext[m_framework.m_config.m_buffers];

		for(uint32_t bCnt = 0; bCnt < m_framework.m_config.m_buffers; ++bCnt)
			m_gfxc[bCnt].init( // Constant Update Engine
				m_framework.m_garlicAllocator.allocate(cueHeapSize,
					Gnm::kAlignmentOfBufferInBytes),
				kNumRingEntries,
				// Draw command buffer
				m_framework.m_onionAllocator.allocate(Gnm::kIndirectBufferMaximumSizeInBytes,
					Gnm::kAlignmentOfBufferInBytes),
				Gnm::kIndirectBufferMaximumSizeInBytes,
				// Constant command buffer
				m_framework.m_onionAllocator.allocate(Gnm::kIndirectBufferMaximumSizeInBytes,
					Gnm::kAlignmentOfBufferInBytes),
				Gnm::kIndirectBufferMaximumSizeInBytes);

		setViewDisplayMode(m_verticalSplit);

		m_sampler.init();
		m_sampler.setMipFilterMode(Gnm::kMipFilterModeLinear);
		m_sampler.setXyFilterMode(Gnm::kFilterModeBilinear, Gnm::kFilterModeBilinear);

		m_vsb = Framework::LoadVsShader(vertexShaderFile, &m_framework.m_allocators);
		m_psb = Framework::LoadPsShader(pixelShaderFile, &m_framework.m_allocators);

		return 0;
	}

	void finalize() override {
		// one GfxContext (command buffer) object is required for each frame that can simultaneously be in flight.
		Gnmx::GfxContext* p_gfxc = &m_gfxc[m_framework.m_backBufferIndex];
		m_framework.terminate(*p_gfxc);
	}

	void update() override {
		for(auto& view : m_views) {
			if(!view.update()) {
				assert(false);
				break;
			}
		}
	}

	void render() override {
		Gnmx::GfxContext* p_gfxc = &(m_gfxc[m_framework.m_backBufferIndex]);

		p_gfxc->reset();
		p_gfxc->initializeDefaultHardwareState();
		m_framework.BeginFrame(*p_gfxc);

		Toolkit::SurfaceUtil::clearRenderTarget(*p_gfxc,
			&m_framework.m_backBuffer->m_renderTarget,
			m_framework.getClearColor());

		Toolkit::SurfaceUtil::clearDepthTarget(*p_gfxc,
			&m_framework.m_backBuffer->m_depthTarget, 1.f);

		p_gfxc->setRenderTargetMask(0x0000000F);
		p_gfxc->setActiveShaderStages(Gnm::kActiveShaderStagesVsPs);
		p_gfxc->setRenderTarget(0, &m_framework.m_backBuffer->m_renderTarget);
		p_gfxc->setDepthRenderTarget(&m_framework.m_backBuffer->m_depthTarget);

		p_gfxc->setupScreenViewport(0, 0, m_framework.m_backBuffer->m_renderTarget.getWidth(),
			m_framework.m_backBuffer->m_renderTarget.getHeight(), 0.5f, 0.5f);

		p_gfxc->setClipRectangle(0, 0, 0, m_framework.m_backBuffer->m_renderTarget.getWidth(),
			m_framework.m_backBuffer->m_renderTarget.getHeight());

		p_gfxc->setPsShader(m_psb);

		p_gfxc->setSamplers(Gnm::kShaderStagePs, 0, 1, &m_sampler);
		p_gfxc->setSamplers(Gnm::kShaderStagePs, 1, 1, &m_sampler);

		p_gfxc->setVsShader(m_vsb, 0, nullptr);

		PsConstants* pPsConstants = static_cast<PsConstants*>(p_gfxc->allocateFromCommandBuffer(sizeof(PsConstants), Gnm::kEmbeddedDataAlignment4));
		if(pPsConstants != 0) {
			Gnm::Buffer constBuffer;

			pPsConstants->hdr_10_10_10_2 = 0;
			pPsConstants->bit_range_coeff = 1.0;
			constBuffer.initAsConstantBuffer(pPsConstants, sizeof(PsConstants));
			p_gfxc->setConstantBuffers(Gnm::kShaderStagePs, 0, 1, &constBuffer);
		}

		for(auto& view : m_views) {
			view.draw(p_gfxc);
		}

		m_framework.EndFrame(*p_gfxc);
	}

	Surface* getSurface(uint32_t index) override {
		assert(index < m_views.size());
		return &m_views[index];
	}

	void setViewDisplayMode(bool splitView) {
		if(splitView) {
			m_views[0].setDisplayMode(View::DisplayMode::SplitLeft);
			m_views[1].setDisplayMode(View::DisplayMode::SplitRight);
		} else {
			m_views[0].setDisplayMode(View::DisplayMode::FullScreen);
			m_views[1].setDisplayMode(View::DisplayMode::Disabled);
		}
	}

	void toggleSplitView() override {
		m_verticalSplit = !m_verticalSplit;
		setViewDisplayMode(m_verticalSplit);
	}
};

MyDisplay::~MyDisplay() {}

MyDisplay* MyDisplay::create() {
	return new MyDisplayImpl();
}
