#pragma once

#include "Twitch.h"
#include "NativePlayer.h"
#include "StreamPlayer.h"

#include <sampleutil.h>
#include <vector>

class Options {
public:
	std::vector<std::string> const& getUrls() const { return m_urls; }
	int getNumFramesToRun() const { return m_numFramesRun; }
	int getSeekTo() const { return m_seekTo; }
	bool getDefaultQuality() const { return m_defaultQuality; }
	bool isDumpSegments() const { return m_dumpSegments; }
	bool isUseFrameLogger() const { return m_useFrameLogger; }
	bool isAbsDisabled() const { return m_disableAbs; }

private:
	std::vector<std::string> m_urls = {
		"https://www.twitch.tv/monstercat", // 24h Live Music
		"https://www.twitch.tv/videos/v205550776", // Short VOD, reported issues with Seek
		"https://www.twitch.tv/videos/93348507", // Bip! Bop!    (Note: The video is actual off by 1 frame) and in mono
		"https://www.twitch.tv/videos/133128389", // Generic VOD from Dark Souls 3
		"https://www.twitch.tv/videos/90812612", // A/V Sync Vod (Bouncing Ball)
		"https://www.twitch.tv/videos/4878428", // Vod with corrupted audio frames, should play fine and skip the bad audio
		"https://www.twitch.tv/videos/258615219", // 7 Audio channels
		"https://twitch.tv/videos/402798647", // VOD with quality change issue
	};
	int m_numFramesRun = 0;
	int m_seekTo = 0;
	bool m_defaultQuality = false;
	bool m_dumpSegments = false;
	bool m_useFrameLogger = false;
	bool m_disableAbs = false;
};

class Timer {
public:
	Timer(float initialTime)
		: m_initialTime(initialTime)
		, m_remaingTime(m_initialTime) {
		assert(m_initialTime >= 0.f);
		assert(m_remaingTime >= 0.f);
	}

	void start(float time) {
		m_initialTime = time;
		m_remaingTime = time;

		assert(m_initialTime >= 0.f);
		assert(m_remaingTime >= 0.f);
	}

	bool update(float dt) {
		assert(dt >= 0.f);

		m_remaingTime -= dt;
		if(m_remaingTime <= 0.f) {
			m_remaingTime = m_initialTime;
			return true;
		}

		return false;
	}

private:
	float m_initialTime;
	float m_remaingTime;
};

class SamplePlayer : public NativePlayer, public StreamPlayer::Listener {
public:
	enum class State {
		Stopped,
		Playing,
		Paused,
	};

	SamplePlayer() { srand(time(nullptr)); }
	int initialize() override;
	int finalize() override;

private:
	Options m_options;
	std::vector<std::unique_ptr<StreamPlayer>> m_players;
	Timer m_stressTestTimer = 1.f;
	uint64_t m_lastTimeCounter = -1;
	State m_state = State::Stopped;
	int m_urlIndex = 0;
	int m_currentFrameCounter = 0;
	float m_dt;
	bool m_multiplePlayers = false;
	bool m_stressTest = false;

	int update() override;
	void onStreamStarted() override;
	void loadCurrentUrl();
	void loadPreviousUrl();
	void loadNextUrl();
	void createPlayers();
	void destroyPlayers();
	void reloadPlayers();
	void seekBackwards(double time);
	void seekForwards(double time);
	void setNextQuality();
	void toggleStressTest();
	void togglePlayPause();
	void toggleSplit();
	void updateControls();
	void updateDebugInfo();
	void handleStressTest();
	void handleReloadStressTest();
	void updateTimeCounter();
};
