#include "pch.h"
#include <libsecure.h>
#include <sce_random.h>
#include <user_service.h>
#include <web_browser_dialog.h>
#include "PS4.h"

#pragma comment(lib, "SceHttp_stub_weak")
#pragma comment(lib, "SceRandom_stub_weak")
#pragma comment(lib, "SceSecure")
#pragma comment(lib, "SceSysmodule_stub_weak")
#pragma comment(lib, "SceUserService_stub_weak")
#pragma comment(lib, "SceCommonDialog_stub_weak")
#pragma comment(lib, "SceWebBrowserDialog_stub_weak")

using namespace Twitch;

BrowserAuth::BrowserAuth() {
#ifndef UNIT_TEST
	// Initialize all components necessary for the Web browser dialog.
	auto result = sceSysmoduleLoadModule(SCE_SYSMODULE_WEB_BROWSER_DIALOG);
	DebugWriteLine("sceSysmoduleLoadModule %#x", result);
	ThrowIfFailed(result);
	result = sceCommonDialogInitialize();
	DebugWriteLine("sceCommonDialogInitialize %#x", result);
	if(result != SCE_COMMON_DIALOG_ERROR_ALREADY_SYSTEM_INITIALIZED) {
		ThrowIfFailed(result);
	}
	SceUserServiceInitializeParams userService{};
	userService.priority = SCE_KERNEL_PRIO_FIFO_DEFAULT;
	result = sceUserServiceInitialize(&userService);
	DebugWriteLine("sceUserServiceInitialize %#x", result);
	if(result != SCE_USER_SERVICE_ERROR_ALREADY_INITIALIZED) {
		ThrowIfFailed(result);
	}
	result = sceWebBrowserDialogInitialize();
	DebugWriteLine("sceWebBrowserDialogInitialize %#x", result);
	if(result != SCE_COMMON_DIALOG_ERROR_ALREADY_INITIALIZED) {
		ThrowIfFailed(result);
	}
#endif
}

#ifdef UNIT_TEST
static bool isRunning;
#endif

BrowserAuth::~BrowserAuth() {
#ifdef UNIT_TEST
	isRunning = false;
#endif
}

std::future<tstring> BrowserAuth::Launch(tstring const& url, tstring const& state, string_t scheme) {
#ifdef UNIT_TEST
	UNREFERENCED_PARAMETER(url);
	UNREFERENCED_PARAMETER(state);
	UNREFERENCED_PARAMETER(scheme);
	if(isRunning) {
		throw TwitchException(FromPlatformError(ERROR_INVALID_OPERATION));
	}
	isRunning = true;
	auto fn = [this] {
		return token;
	};
#else
	// Ensure the Web browser dialog isn't already open.
	auto const result = sceWebBrowserDialogUpdateStatus();
	if(result == SCE_COMMON_DIALOG_STATUS_RUNNING) {
		throw TwitchException(FromPlatformError(ERROR_INVALID_OPERATION));
	}

	// Launch the browser to start the OAuth 2 flow to have the user
	// authenticate.  For client authentication, there will be a redirect URI
	// with a custom scheme.
	SceWebBrowserDialogParam param;
	sceWebBrowserDialogParamInitialize(&param);
	param.mode = SCE_WEB_BROWSER_DIALOG_MODE_DEFAULT;
	param.url = url.c_str();
	SceWebBrowserDialogCallbackInitParam callbackParam{};
	if(scheme != nullptr) {
		callbackParam.size = sizeof(callbackParam);
		callbackParam.type = SCE_WEB_BROWSER_DIALOG_CALLBACK_PARAM_TYPE_URL;
		callbackParam.data = scheme;
		param.callbackInitParam = &callbackParam;
	}
	ThrowIfFailed(sceUserServiceGetInitialUser(&param.userId));
	ThrowIfFailed(sceWebBrowserDialogOpen(&param));
	bool isCustomScheme = scheme != nullptr;
	auto fn = [this, state, isCustomScheme] {
		for(;;) {
			auto result = sceWebBrowserDialogUpdateStatus();
			if(result == SCE_COMMON_DIALOG_STATUS_FINISHED) {
				if(isCustomScheme) {
					SceWebBrowserDialogResult resultParam{};
					SceWebBrowserDialogCallbackResultParam callbackParam{};
					callbackParam.size = sizeof(callbackParam);
					resultParam.callbackResultParam = &callbackParam;
					ThrowIfFailed(sceWebBrowserDialogGetResult(&resultParam));
					auto returnedState = ExtractToken(resultParam.callbackResultParam->data, "state");
					if(returnedState == state) {
						auto s = ExtractToken(resultParam.callbackResultParam->data);
						if(s.empty()) {
							s = ExtractToken(resultParam.callbackResultParam->data, "access_token");
						}
						TokenLock lock(tokenAccessor);
						token = s;
					}
				}
				break;
			}
			sceKernelUsleep(110 * 1000);
		}
		ThrowIfFailed(sceWebBrowserDialogTerminate());
		TokenLock lock(tokenAccessor);
		return token;
	};
#endif
	return std::async(std::launch::deferred, fn);
}

void BrowserAuth::Stop(tstring const& token_) {
#ifdef UNIT_TEST
	isRunning = false;
	token = token_;
#else
	// Set the token and close the Web browser dialog.
	TokenLock lock(tokenAccessor);
	token = token_;
	lock.unlock();
	sceWebBrowserDialogClose();
#endif
}
