#include "..\Shared\Twitch.h"
#include "pch.h"
#include "mediaplayer/platforms/ps4/playercore.hpp"
#include "../platforms/ps4/videoframe.hpp"

#pragma comment(lib, "SceAppContent_stub_weak")
#pragma comment(lib, "SceAudiodec_stub_weak")
#pragma comment(lib, "SceAudioOut_stub_weak")
#pragma comment(lib, "ScePosix_stub_weak")
#pragma comment(lib, "SceVideodec2_stub_weak")
#ifdef _DEBUG
# pragma comment(lib, "../vendor/player-core/build/ps4/6.500/lib/Debug/playercored.a")
# pragma comment(lib, "../vendor/player-core/build/ps4/6.500/external/lib/captiond.a")
# pragma comment(lib, "../vendor/player-core/build/ps4/6.500/external/lib/json11d.a")
#else
# pragma comment(lib, "../vendor/player-core/build/ps4/6.500/lib/Release/playercore.a")
# pragma comment(lib, "../vendor/player-core/build/ps4/6.500/external/lib/caption.a")
# pragma comment(lib, "../vendor/player-core/build/ps4/6.500/external/lib/json11.a")
#endif

using namespace Twitch;

#include "ListenerImpl.inl"

namespace {
	twitch::MediaTime FromDouble(double time) {
		return time == std::numeric_limits<double>::infinity() ? twitch::MediaTime::max() : twitch::MediaTime(time);
	}

	double ToDouble(twitch::MediaTime const& time) {
		return time == twitch::MediaTime::max() ? std::numeric_limits<double>::infinity() : time.seconds();
	}
}

struct Player::PlayerImpl {
	PlayerImpl(Observer& observer) : listener(observer) {
		if(!twitch::ps4::PlayerCore::isInitialized()) {
			twitch::ps4::Configuration config;
			config.deviceId = FromTstring(GetDeviceId());
			if(!twitch::ps4::PlayerCore::initialize(config)) {
				throw std::runtime_error("PlayerCore::initialize");
			}
		}
		auto onVideoFrameBuffer = [this](std::shared_ptr<twitch::VideoFrameBuffer> const& videoFrameBuffer_) {
			std::lock_guard<std::mutex> lock(mutex);
			videoFrameBuffer = videoFrameBuffer_;
		};
		auto onFrameNeedsReleasing = [](const std::shared_ptr<twitch::VideoFrame>&) {};
		player = twitch::ps4::PlayerCore::create(listener, onVideoFrameBuffer, onFrameNeedsReleasing);
	}

	bool Access(Player::AccessFn fn) {
		std::lock_guard<std::mutex> lock(mutex);
		if(videoFrameBuffer) {
			auto videoFrame = videoFrameBuffer->getFrame();
			if(videoFrame) {
				if(fn(videoFrame->getOutputInfo(), videoFrame->getPictureInfo())) {
					videoFrame->finish();
				}
			}
			return true;
		}
		return false;
	}

	ListenerImpl listener;
	std::mutex mutex;
	std::shared_ptr<twitch::VideoFrameBuffer> videoFrameBuffer;
	std::shared_ptr<twitch::Player> player;
};

Player::Player(Observer& observer) {
	pimpl = std::make_unique<PlayerImpl>(observer);
}

bool Twitch::Player::Access(AccessFn fn) {
	return pimpl->Access(fn);
}

Player::Player(Player&&) noexcept = default;
Player& Player::operator=(Player&&) noexcept = default;

Player::~Player() {}

void Player::Load(tstring const& url) {
	pimpl->player->load(FromTstring(url));
}

void Player::Play() {
	pimpl->player->play();
}

void Player::Pause() {
	pimpl->player->pause();
}

bool Player::GetIsSeekable() const {
	return pimpl->player->isSeekable();
}

double Player::GetDuration() const {
	return ToDouble(pimpl->player->getDuration());
}

double Player::GetPosition() const {
	return ToDouble(pimpl->player->getPosition());
}

void Player::SetPosition(double position) {
	pimpl->player->seekTo(FromDouble(position));
}

double Player::GetBufferedPosition() const {
	return ToDouble(pimpl->player->getBufferedPosition());
}

Player::PlayerState Player::GetState() const {
	return static_cast<PlayerState>(pimpl->player->getState());
}

float Player::GetPlaybackRate() const {
	return pimpl->player->getPlaybackRate();
}

void Player::SetPlaybackRate(float playbackRate) {
	pimpl->player->setPlaybackRate(playbackRate);
}

bool Player::GetIsLooping() const {
	return pimpl->player->isLooping();
}

void Player::SetIsLooping(bool isLooping) {
	pimpl->player->setLooping(isLooping);
}

bool Player::GetIsMuted() const {
	return pimpl->player->isMuted();
}

void Player::SetIsMuted(bool isMuted) {
	pimpl->player->setMuted(isMuted);
}

float Player::GetVolume() const {
	return pimpl->player->getVolume();
}

void Player::SetVolume(float volume) {
	pimpl->player->setVolume(volume);
}
