#include "pch.h"
#include "RawSocket.h"

#pragma comment(lib, "libSceNet_stub_weak.a")

using namespace Twitch;

constexpr std::chrono::seconds Socket::DefaultTimeout /*= std::chrono::seconds(10)*/;

namespace {
	int ResolveAddress(const char *hostname, SceNetInAddr *addr) noexcept {
		int result = 0;
		auto memid = sceNetPoolCreate("sceNetPoolCreate", 4 * 1024, 0);
		if(memid >= 0) {
			auto rid = sceNetResolverCreate("sceNetResolverCreate", memid, 0);
			if(rid >= 0) {
				result = sceNetResolverStartNtoa(rid, hostname, addr, 0, 0, 0);
				if(result >= 0) {
					result = 0;
				} else {
					DebugWriteLine("[ResolveAddress::sceNetResolverStartNtoa] error %#x (%d)\n", result, sce_net_errno);
				}
				sceNetResolverDestroy(rid);
			} else {
				DebugWriteLine("[ResolveAddress::sceNetResolverCreate] error %#x (%d)\n", rid, sce_net_errno);
				result = rid;
			}
			sceNetPoolDestroy(memid);
		} else {
			DebugWriteLine("[ResolveAddress::sceNetPoolCreate] error %#x (%d)\n", memid, sce_net_errno);
			result = memid;
		}
		return result;
	}
}

RawSocket::RawSocket() {
	value = sceNetSocket("Twitch::sceNetSocket", SCE_NET_AF_INET, SCE_NET_SOCK_STREAM, SCE_NET_IPPROTO_IP);
	if(value < 0) {
		throw TwitchException(value);
	}
}

RawSocket::~RawSocket() {
	Close();
}

void RawSocket::InternalClose() noexcept {
	auto value_ = __sync_swap(&value, 0);
	if(value_) {
		sceNetSocketAbort(value_, 0);
		sceNetSocketClose(value_);
	}
}

int RawSocket::InternalConnect(string_t hostName, unsigned short port, std::chrono::milliseconds timeout) noexcept {
	auto timeout_ = static_cast<int>(timeout.count() * 1000);
	int errorCode = sceNetSetsockopt(value, SCE_NET_SOL_SOCKET, SCE_NET_SO_CONNECTTIMEO, &timeout_, sizeof(timeout_));
	if(errorCode < 0) {
		DebugWriteLine(_T("[RawSocket::InternalConnect] sceNetSetsockopt(SCE_NET_SO_CONNECTTIMEO) failed %#x (%d)"),
			errorCode, sce_net_errno);
		return errorCode;
	}
	SceNetSockaddrIn sa{};
	if(sceNetInetPton(SCE_NET_AF_INET, hostName, &sa.sin_addr) <= 0) {
		errorCode = ResolveAddress(hostName, &sa.sin_addr);
		if(errorCode < 0) {
			return errorCode;
		}
	}
	sa.sin_family = SCE_NET_AF_INET;
	sa.sin_port = sceNetHtons(port);
	errorCode = sceNetConnect(value, reinterpret_cast<SceNetSockaddr const*>(&sa), sizeof(sa));
	if(errorCode < 0) {
		DebugWriteLine(_T("[RawSocket::InternalConnect] sceNetConnect failed %#x (%d)"), errorCode, sce_net_errno);
		return errorCode;
	}
	return 0;
}

int RawSocket::InternalReceive(void* buffer, size_t size) noexcept {
	return sceNetRecv(value, buffer, size, 0);
}

int RawSocket::InternalSend(void const* buffer, size_t size) noexcept {
	return sceNetSend(value, buffer, size, 0);
}
