#include "pch.h"
#include "HttpUtil.h"
#include "Internal.h"

using namespace std::literals;
using namespace Twitch;

namespace {
	constexpr auto apiPlatformUrlRoot = _T("https://api.twitch.tv/api/platform/");
	constexpr auto pollingInterval = 3s;
}

tstring ActivationAuth::GenerateActivationToken() {
	// Generate forty random characters.
	tstring activationToken;
	std::generate_n(std::back_inserter(activationToken), 40, []() {
		unsigned int value;
		rand_s(&value);
		return static_cast<tstring::value_type>(((value & 0x6f) | 0x40) + 1);
	});
	return activationToken;
}

std::future<tstring> ActivationAuth::StartAuth(tstring const& applicationId, tstring const& activationToken) {
	// Validate the arguments.
	if(applicationId.empty()) {
		DebugWriteLine(_T("[ActivationAuth::StartAuth] Invalid application identifier"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}
	if(activationToken.empty()) {
		DebugWriteLine(_T("[ActivationAuth::StartAuth] Invalid activation token"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}

	auto url = apiPlatformUrlRoot + applicationId + _T("/generate?token=") + UrlEncode(activationToken);
	auto fn = [url] {
		// Submit the token generation request.
		auto httpResponse = HttpRequest().Post(url);
		ThrowIfResponseFailed(httpResponse);

		// Parse the response as JSON.
		auto const& response = httpResponse.Response;
		std::string parseError;
		auto jsonVal = json11::Json::parse(std::string(response.cbegin(), response.cend()).c_str(), parseError);
		if(!parseError.empty()) {
			DebugWriteLine(_T("[ActivationAuth::StartAuth] JSON parsing failed:  %s"), ToTstring(parseError).c_str());
			throw TwitchException(FromPlatformError(ERROR_INVALID_RESPONSE));
		}

		// Return the entry code.
		if(jsonVal.is_object()) {
			auto const& jsonCode = jsonVal["code"];
			if(jsonCode.is_string()) {
				return ToTstring(jsonCode.string_value());
			}
		}
		DebugWriteLine(_T("[ActivationAuth::StartAuth] unexpected JSON response"));
		throw TwitchException(FromPlatformError(ERROR_INVALID_RESPONSE));
	};
	return std::async(std::launch::deferred, fn);
}

std::future<tstring> ActivationAuth::AcquireAccessToken(tstring const& applicationId, tstring const& activationToken) {
	// Validate the arguments.
	if(applicationId.empty()) {
		DebugWriteLine(_T("[ActivationAuth::AcquireAccessToken] Invalid application identifier"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}
	if(activationToken.empty()) {
		DebugWriteLine(_T("[ActivationAuth::AcquireAccessToken] Invalid activation token"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}

	isCanceled = false;
	auto url = apiPlatformUrlRoot + applicationId + _T("/authenticate?token=") + UrlEncode(activationToken);
	auto fn = [this, url] {
		while(!isCanceled) {
			// Poll for a token.
			auto httpResponse = HttpRequest().Post(url);
			if(httpResponse.Result) {
				throw TwitchException(httpResponse.Result);
			}

			auto resultCode = httpResponse.ResultCode;
			if(resultCode == 200) {
				// Parse the response as JSON.
				auto const& response = httpResponse.Response;
				std::string parseError;
				auto jsonVal = json11::Json::parse(std::string(response.cbegin(), response.cend()).c_str(), parseError);
				if(!parseError.empty()) {
					DebugWriteLine(_T("[ActivationAuth::AcquireAccessToken] JSON parsing failed:  %s"), ToTstring(parseError).c_str());
					throw TwitchException(FromPlatformError(ERROR_INVALID_RESPONSE));
				}

				// Return the OAuth token.
				if(jsonVal.is_object()) {
					auto const& jsonToken = jsonVal["oauth_token"];
					if(jsonToken.is_string()) {
						return ToTstring(jsonToken.string_value());
					}
				}
				DebugWriteLine(_T("[ActivationAuth::AcquireAccessToken] unexpected JSON response"));
				throw TwitchException(FromPlatformError(ERROR_INVALID_RESPONSE));
			}

			// Try again later.
			std::this_thread::sleep_for(pollingInterval);
		}

		// Return an empty token upon cancelation.
		return tstring();
	};
	return std::async(std::launch::deferred, fn);
}

void ActivationAuth::Cancel() {
	isCanceled = true;
}

std::future<void> Twitch::ActivationAuth::SignOut(tstring const& applicationId, tstring const& accessToken) {
	// Validate the arguments.
	if(applicationId.empty()) {
		DebugWriteLine(_T("[ActivationAuth::SignOut] Invalid application identifier"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}
	if(accessToken.empty()) {
		DebugWriteLine(_T("[ActivationAuth::SignOut] Invalid access token"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}

	auto url = apiPlatformUrlRoot + applicationId + _T("/disconnect?oauth_token=") + UrlEncode(accessToken);
	auto fn = [url] {
		// Submit the sign-out request.
		auto httpResponse = HttpRequest().Post(url);
		ThrowIfResponseFailed(httpResponse);
	};
	return std::async(std::launch::deferred, fn);
}
