#include "version.inl"

using namespace std::string_literals;
using namespace Twitch;

namespace {
	struct ScienceEvent {
		using vector = std::vector<ScienceEvent>;

		tstring clientId, token, userId, method;
		stringmap properties;
	};

	std::atomic<ScienceEvent::vector*> events;
	std::promise<void> stop_event;
	tstring const spadeUrl = _T("https://spade.twitch.tv/");
	tstring currentToken;
	std::string applicationName, deviceId, currentUserId;
	std::future<void> event_task;
	tstring const empty;

	std::string EncodeProperties(stringmap const& properties) {
		std::map<std::string, std::string> jsonProperties;
		for(auto const& property : properties) {
			auto key = std::string(property.first.cbegin(), property.first.cend());
			auto value = std::string(property.second.cbegin(), property.second.cend());
			jsonProperties.insert({ key, value });
		}
		auto jsonSdkPayload = json11::Json(jsonProperties);
		return jsonSdkPayload.dump();
	}

	void SendScienceEvent(ScienceEvent const& scienceEvent) {
		// Get the user identifier, if available or necessary.
		std::string userId;
		if(!scienceEvent.userId.empty()) {
			userId = FromTstring(scienceEvent.userId);
		} else if(!scienceEvent.token.empty()) {
			if(scienceEvent.token != currentToken) {
				// Fetch the user for its identifier.
				currentUserId = FromTstring(User::FetchCurrent(scienceEvent.clientId, scienceEvent.token).get().Id);
			}
			userId = currentUserId;
		}

		// Create a JSON payload for science.
		std::map<std::string, json11::Json> value;
		value["event"] = "sdk_request";
		value["properties"] = json11::Json(std::map<std::string, json11::Json>{
			{ "client_id"s, FromTstring(scienceEvent.clientId) },
			{ "sdk_method"s, FromTstring(scienceEvent.method) },
			{ "user_id"s, userId },
			{ "sdk_version"s, sdkVersion },
			{ "platform"s, sdkPlatform },
			{ "runtime"s, sdkRuntime },
			{ "application_name"s, applicationName },
			{ "device_id"s, deviceId },
			{ "sdk_json_payload"s, EncodeProperties(scienceEvent.properties) },
		});

		// Format the payload as a base64-encoded POST request and send it to Spade.
		std::string json = json11::Json(value).dump();
		std::string data = Base64Encode(std::vector<char>(json.cbegin(), json.cend()));
		auto const httpResponse = HttpRequest().Post(spadeUrl, std::vector<char>(data.cbegin(), data.cend()));
#ifdef _DEBUG
		int const result = httpResponse.Result;
		if(result == 0) {
			auto resultCode = httpResponse.ResultCode;
			auto responseBody = httpResponse.Response;
			UNREFERENCED_PARAMETER(resultCode);
			UNREFERENCED_PARAMETER(responseBody);
		}
#else
		UNREFERENCED_PARAMETER(httpResponse);
#endif
	}

	void SendScienceEvents() {
		auto check_stop = stop_event.get_future();
		do {
			std::unique_ptr<ScienceEvent::vector> p(events.exchange(nullptr));
			if(p) {
				for(auto const& e : *p) {
					try {
						SendScienceEvent(e);
					} catch(std::exception const& ex) {
						DebugWriteLine(_T("SendScienceEvents failed: %s"), ToTstring(ex.what()).c_str());
						UNREFERENCED_PARAMETER(ex);
					}
				}
			}
		} while(check_stop.wait_for(std::chrono::seconds(5)) != std::future_status::ready);
	}

	void StopScienceEvents() {
		stop_event.set_value();
	}

	void SendScienceEvent(tstring const& clientId, tstring const& token, tstring const& userId, tstring const& method, stringmap const& properties) noexcept {
		try {
			std::unique_ptr<ScienceEvent::vector> p(events.exchange(nullptr));
			if(p == nullptr) {
				p.reset(new ScienceEvent::vector);
			}
			p->push_back({ clientId, token, userId, method, properties });
			events.exchange(p.release());
			if(!event_task.valid()) {
				applicationName = FromTstring(GetApplicationName());
				deviceId = FromTstring(GetDeviceId());
				event_task = std::async(std::launch::async, SendScienceEvents);
				atexit(StopScienceEvents);
			}
		} catch(std::exception const& ex) {
			DebugWriteLine(_T("warning: SendScienceEvent failed: %s"), ToTstring(ex.what()).c_str());
			UNREFERENCED_PARAMETER(ex);
			// Ignore exceptions.
		} catch(...) {
			DebugWriteLine(_T("warning: SendScienceEvent failed"));
			// Ignore exceptions.
		}
	}
}

void Twitch::SendTokenScienceEvent(tstring const& clientId, tstring const& token, tstring const& method, stringmap const& properties) noexcept {
	SendScienceEvent(clientId, token, empty, method, properties);
}

void Twitch::SendUserScienceEvent(tstring const& clientId, tstring const& userId, tstring const& method, stringmap const& properties) noexcept {
	SendScienceEvent(clientId, empty, userId, method, properties);
}
