#include "pch.h"
#include "Http.h"
#include "Internal.h"

using namespace Twitch;

namespace {
	tstring const helixUrl = _T("https://api.twitch.tv/helix/");
}

std::future<tstring> Clip::Create(tstring const& clientId, tstring const& token, tstring const& broadcasterId, bool hasDelay /*= false*/) {
	// Validate the arguments.
	if(clientId.empty()) {
		DebugWriteLine(_T("[Clip::Create] Invalid client identifier"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}
	if(token.empty()) {
		DebugWriteLine(_T("[Clip::Create] Invalid token"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}
	if(broadcasterId.empty()) {
		DebugWriteLine(_T("[Clip::Create] Invalid broadcaster identifier"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}

	auto fn = [clientId, token, broadcasterId, hasDelay] {
		// Create the requested clip.
		auto url = helixUrl + _T("clips?broadcaster_id=") + broadcasterId;
		if(hasDelay) {
			url += _T("has_delay=true");
		}
		auto httpResponse = HttpRequest(clientId, token).Post(url);
		ThrowIfResponseFailed(httpResponse);

		// Parse the response as JSON.
		auto const& response = httpResponse.Response;
		std::string parseError;
		auto jsonVal = json11::Json::parse(std::string(response.cbegin(), response.cend()).c_str(), parseError);
		if(!parseError.empty()) {
			DebugWriteLine(_T("[Clip::Create] JSON parsing failed:  %s"), ToTstring(parseError).c_str());
			throw TwitchException(FromPlatformError(ERROR_INVALID_RESPONSE));
		}

		// Send a science event.
		SendTokenScienceEvent(clientId, token, _T("sdk_clip_create"), { { _T("broadcaster_id"), broadcasterId } });

		// Return the identifier of the created clip.
		auto const& jsonClip = jsonVal["data"][0];
		auto const& jsonId = jsonClip["id"];
		return ToTstring(jsonId.string_value());
	};
	return std::async(std::launch::deferred, fn);
}

std::future<Clip> Clip::Fetch(tstring const& clientId, tstring const& id) {
	// Validate the arguments.
	if(clientId.empty()) {
		DebugWriteLine(_T("[Clip::Fetch] Invalid client identifier"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}
	if(id.empty()) {
		DebugWriteLine(_T("[Clip::Fetch] Invalid clip identifier"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}

	auto fn = [clientId, id] {
		// Fetch the requested clip.
		auto url = helixUrl + _T("clips?id=") + id;
		auto httpResponse = HttpRequest(clientId).Get(url);
		ThrowIfResponseFailed(httpResponse);

		// Parse the response as JSON.
		auto const& response = httpResponse.Response;
		std::string parseError;
		auto jsonVal = json11::Json::parse(std::string(response.cbegin(), response.cend()).c_str(), parseError);
		if(!parseError.empty()) {
			DebugWriteLine(_T("[Clip::Fetch] JSON parsing failed:  %s"), ToTstring(parseError).c_str());
			throw TwitchException(FromPlatformError(ERROR_INVALID_RESPONSE));
		}

		// Send a science event.
		SendUserScienceEvent(clientId, tstring(), _T("sdk_clip_get"), { { _T("clip_id"), id } });

		// Get references for all of the clip fields.
		auto const& jsonClip = jsonVal["data"][0];
		auto const& jsonId = jsonClip["id"];
		auto const& jsonUrl = jsonClip["url"];
		auto const& jsonEmbedUrl = jsonClip["embed_url"];
		auto const& jsonBroadcasterId = jsonClip["broadcaster_id"];
		auto const& jsonBroadcasterLogin = jsonClip["broadcaster_name"];
		auto const& jsonCreatorId = jsonClip["creator_id"];
		auto const& jsonCreatorLogin = jsonClip["creator_name"];
		auto const& jsonVideoId = jsonClip["video_id"];
		auto const& jsonGameId = jsonClip["game_id"];
		auto const& jsonLanguage = jsonClip["language"];
		auto const& jsonTitle = jsonClip["title"];
		auto const& jsonViewCount = jsonClip["view_count"];
		auto const& jsonCreatedAt = jsonClip["created_at"];
		auto const& jsonThumbnailUrl = jsonClip["thumbnail_url"];

		// Populate and return a Clip instance.
		Clip clip;
		clip.id = ToTstring(jsonId.string_value());
		clip.url = ToTstring(jsonUrl.string_value());
		clip.embedUrl = ToTstring(jsonEmbedUrl.string_value());
		clip.broadcasterId = ToTstring(jsonBroadcasterId.string_value());
		clip.broadcasterLogin = ToTstring(jsonBroadcasterLogin.string_value());
		clip.creatorId = ToTstring(jsonCreatorId.string_value());
		clip.creatorLogin = ToTstring(jsonCreatorLogin.string_value());
		clip.videoId = ToTstring(jsonVideoId.string_value());
		clip.gameId = ToTstring(jsonGameId.string_value());
		clip.language = ToTstring(jsonLanguage.string_value());
		clip.title = ToTstring(jsonTitle.string_value());
		clip.viewCount = static_cast<unsigned>(jsonViewCount.int_value());
		clip.createdAt = ToTstring(jsonCreatedAt.string_value());
		clip.thumbnailUrl = ToTstring(jsonThumbnailUrl.string_value());
		return clip;
	};
	return std::async(std::launch::deferred, fn);
}
