#include "pch.h"
#include "Http.h"
#include "Internal.h"
#define LOCAL_WS_SERVER

using namespace std::literals;
using namespace Twitch;

namespace {
#ifdef LOCAL_WS_SERVER
# if defined(_XBOX_ONE) || defined(__ORBIS__)
#  include "../ws_url.inl"
# else
	constexpr string_t url = _T("ws://localhost:6006");
# endif
#else
	constexpr string_t url = _T("wss://metadata.twitch.tv/api/ingest");
#endif
	std::basic_regex<tstring::value_type> nameRx(_T("^\\w+(\\.\\w+|\\[[0-9]\\])*$"), std::regex_constants::ECMAScript);

	enum class MessageType { None, Append, Reauthorize, Reconnect, Refresh, Remove, Update };

	struct Message {
		MessageType type = MessageType::None;
		std::string name;
		json11::Json value;

		json11::Json to_json() const {
			json11::Json::array array;
			switch(type) {
			case MessageType::Append:
				array.push_back(name);
				array.push_back("a");
				array.push_back(value);
				break;
			case MessageType::Remove:
				array.push_back(name);
				break;
			case MessageType::Update:
				array.push_back(name);
				array.push_back(value);
				break;
			default:
				assert(false);
			}
			return array;
		}
	};

	using state_t = json11::Json::object;

	class SynchronizedState {
	public:
		SynchronizedState() = default;
		SynchronizedState(SynchronizedState const&) = delete;
		SynchronizedState(SynchronizedState&&) = default;
		SynchronizedState& operator=(SynchronizedState const&) = delete;
		SynchronizedState& operator=(SynchronizedState&&) = default;
		~SynchronizedState() = default;

		template<typename FN>
		void Access(FN fn) {
			std::unique_lock<decltype(mutex)> lock(mutex);
			fn(currentState, queue);
		}

	private:
		state_t currentState;
		std::mutex mutex;
		std::deque<Message> queue;
	};

	std::string generateSessionId() {
		std::string sessionId;
		std::generate_n(std::back_inserter(sessionId), 32, [] {
			unsigned int value;
			rand_s(&value);
			return static_cast<std::string::value_type>(((value & 0x6f) | 0x40) + 1);
		});
		return sessionId;
	}

	void CorrectMetadata(state_t& state, json11::Json sessionId = generateSessionId(), json11::Json isActive = true) {
		// Add the "_metadata" field if not present.  If it is present, ensure it
		// contains the required fields of the expected types.
		auto it = state.find("_metadata");
		if(it == state.cend()) {
			state.insert({ "_metadata", state_t{
				{ "id", sessionId },
				{ "active", isActive },
			} });
		} else {
			if(!it->second.is_object()) {
				DebugWriteLine(_T("[CorrectMetadata] _metadata value is not an object"));
				throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
			}
			auto metadata = it->second.object_items();
			it = metadata.find("id");
			if(it == metadata.cend()) {
				metadata.insert({ "id", sessionId });
			} else if(!it->second.is_string()) {
				DebugWriteLine(_T("[CorrectMetadata] _metadata.id value is not a string"));
				throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
			}
			it = metadata.find("active");
			if(it == metadata.cend()) {
				metadata.insert({ "active", true });
			} else if(!it->second.is_bool()) {
				DebugWriteLine(_T("[CorrectMetadata] _metadata.active value is not a Boolean"));
				throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
			}
			state["_metadata"] = metadata;
		}
	}

	std::string const accessors = ".[";

	json11::Json UpdateState(json11::Json state, std::string const& name, std::function<json11::Json(json11::Json, std::string const&)> fn) {
		auto i = name.find_first_of(accessors, 1);
		if(i == name.npos) {
			return fn(state, name);
		} else if(name.front() == '[') {
			// This is an array field access.
			if(!state.is_array()) {
				DebugWriteLine(_T("[UpdateState] \"%s\" is not an array"), ToTstring(name).c_str());
				throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
			}
			auto array = state.array_items();
			auto index = std::stoul(name.substr(1));
			if(index >= array.size()) {
				DebugWriteLine(_T("[UpdateState] array index %d is out of bounds (%d) for \"%s\""),
					index, array.size(), ToTstring(name).c_str());
				throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
			}
			auto rest = name.substr(name.find(']') + (name[i] == '.' ? 2 : 1));
			array[index] = UpdateState(array[index], rest, fn);
			return array;
		} else {
			// This is an object field access.
			if(!state.is_object()) {
				DebugWriteLine(_T("[UpdateState] \"%s\" is not an object"), ToTstring(name).c_str());
				throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
			}
			auto object = state.object_items();
			auto first = name.substr(0, i);
			auto it = object.find(first);
			if(it == object.cend()) {
				DebugWriteLine(_T("[UpdateState] \"%s\" does not specify a known field"), ToTstring(name).c_str());
				throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
			}
			auto rest = name.substr(i + (name[i] == '.' ? 1 : 0));
			it->second = UpdateState(it->second, rest, fn);
			return object;
		}
	}
}

tstring const DataSource::EmptyState = _T("{}");

#include "DataSourceImpl.inl"

DataSource::DataSource() = default;
DataSource::DataSource(DataSource&&) noexcept = default;
DataSource& DataSource::operator=(DataSource&&) noexcept = default;
DataSource::~DataSource() = default;

void DataSource::Connect(Configuration const& configuration) {
	// Ensure Connect has not already been invoked.
	if(pimpl && pimpl->IsValid) {
		DebugWriteLine(_T("[DataSource::Connect] this already connected"));
		throw TwitchException(FromPlatformError(ERROR_INVALID_STATE));
	}
	pimpl = std::make_unique<DataSourceImpl>(configuration);
}

void DataSource::Disconnect() {
	pimpl = nullptr;
}

void DataSource::ReplaceState(tstring const& state) {
	// Ensure Connect has already been invoked.
	if(!pimpl || !pimpl->IsValid) {
		DebugWriteLine(_T("[DataSource::ReplaceState] connection not established"));
		throw TwitchException(FromPlatformError(ERROR_INVALID_STATE));
	}

	// Validate the state.
	std::string parseError;
	auto json = json11::Json::parse(FromTstring(state), parseError);
	if(!parseError.empty()) {
		DebugWriteLine(_T("[DataSource::ReplaceState] JSON parse error:  %s"), ToTstring(parseError).c_str());
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}
	if(!json.is_object()) {
		DebugWriteLine(_T("[DataSource::ReplaceState] state is not a JSON object"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}
	auto newState = json.object_items();

	pimpl->Access([&newState](auto& currentState, auto& queue) {
		// Merge required metadata into the state, if necessary.
		auto metadata = currentState["_metadata"];
		CorrectMetadata(newState, metadata["id"], metadata["active"]);

		// Update the current state.
		currentState = newState;

		// Queue the "Refresh" message.
		queue.push_back({ MessageType::Refresh, "", "" });
	});
}

void DataSource::RemoveField(tstring const& name) {
	// Ensure Connect has already been invoked.
	if(!pimpl || !pimpl->IsValid) {
		DebugWriteLine(_T("[DataSource::RemoveField] connection not established"));
		throw TwitchException(FromPlatformError(ERROR_INVALID_STATE));
	}

	// Validate the name.
	if(name.empty()) {
		DebugWriteLine(_T("[DataSource::RemoveField] name is empty"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	} else if(name.back() == ']') {
		DebugWriteLine(_T("[DataSource::RemoveField] \"%s\" does not specify a field"), name.c_str());
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	} else if(name == _T("_metadata") || name == _T("_metadata.active") || name == _T("_metadata.id")) {
		DebugWriteLine(_T("[DataSource::RemoveField] cannot remove \"%s\""), name.c_str());
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	} else if(!std::regex_search(name, nameRx)) {
		DebugWriteLine(_T("[DataSource::RemoveField] \"%s\" is not a valid field specifier"), name.c_str());
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}

	// Update the current state.
	pimpl->Access([name = FromTstring(name)](auto& currentState, auto& queue) {
		currentState = UpdateState(currentState, name, [](json11::Json state, std::string const& name) {
			auto object = state.object_items();
			if(!object.erase(name)) {
				DebugWriteLine(_T("[DataSource::RemoveField] \"%s\" is not a valid field specifier"), ToTstring(name).c_str());
				throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
			}
			return object;
		}).object_items();

		// Queue the "Remove Field" message.
		queue.push_back({ MessageType::Remove, name, "" });
	});
}

void DataSource::UpdateField(tstring const& name, tstring const& value_) {
	// Ensure Connect has already been invoked.
	if(!pimpl || !pimpl->IsValid) {
		DebugWriteLine(_T("[DataSource::UpdateField] connection not established"));
		throw TwitchException(FromPlatformError(ERROR_INVALID_STATE));
	}

	// Validate the name and value.
	if(name.empty()) {
		DebugWriteLine(_T("[DataSource::UpdateField] name is empty"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	} else if(!std::regex_search(name, nameRx)) {
		DebugWriteLine(_T("[DataSource::RemoveField] \"%s\" is not a valid field specifier"), name.c_str());
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}
	std::string parseError;
	auto value = json11::Json::parse(FromTstring(value_), parseError);
	if(!parseError.empty()) {
		DebugWriteLine(_T("[DataSource::UpdateField] JSON parse error:  %s"), ToTstring(parseError).c_str());
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	} else if(name == _T("_metadata") && !value.is_object()) {
		DebugWriteLine(_T("[DataSource::UpdateField] _metadata update value is not a JSON object"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	} else if(name == _T("_metadata.active") && !value.is_bool()) {
		DebugWriteLine(_T("[DataSource::UpdateField] _metadata.active update value is not Boolean"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	} else if(name == _T("_metadata.id") && !value.is_string()) {
		DebugWriteLine(_T("[DataSource::UpdateField] _metadata update value is not a string"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}

	pimpl->Access([name = FromTstring(name), &value](auto& currentState, auto& queue) {
		// Merge required metadata into the state, if necessary.
		if(name == "_metadata") {
			auto newState = std::remove_reference_t<decltype(currentState)>{ { "_metadata", value.object_items() } };
			auto metadata = currentState["_metadata"];
			CorrectMetadata(newState, metadata["id"], metadata["active"]);
			value = newState["_metadata"];
		}

		// Update the current state.
		currentState = UpdateState(currentState, name, [&value](json11::Json state, std::string const& name) -> json11::Json {
			if(name.front() == '[') {
				// This is an array field access.
				auto array = state.array_items();
				auto index = std::stoul(name.substr(1));
				if(index >= array.size()) {
					DebugWriteLine(_T("[DataSource::UpdateField] array index %d is out of bounds (%d) for \"%s\""),
						index, array.size(), ToTstring(name).c_str());
					throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
				}
				array[index] = value;
				return array;
			} else {
				auto object = state.object_items();
				object[name] = value;
				return object;
			}
		}).object_items();

		// Queue the "Update Field" message.
		queue.push_back({ MessageType::Update, name, value });
	});
}

void DataSource::AppendToArrayField(tstring const& name, tstring const& value_) {
	// Ensure Connect has already been invoked.
	if(!pimpl || !pimpl->IsValid) {
		DebugWriteLine(_T("[DataSource::AppendToArrayField] connection not established"));
		throw TwitchException(FromPlatformError(ERROR_INVALID_STATE));
	}

	// Validate the name and value.
	if(name.empty()) {
		DebugWriteLine(_T("[DataSource::AppendToArrayField] name is empty"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}
	std::string parseError;
	auto value = json11::Json::parse(FromTstring(value_), parseError);
	if(!parseError.empty()) {
		DebugWriteLine(_T("[DataSource::AppendToArrayField] JSON parse error:  %s"), ToTstring(parseError).c_str());
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	} else if(!value.is_array()) {
		DebugWriteLine(_T("[DataSource::AppendToArrayField] value is not an array"));
		throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
	}
	auto array = value.array_items();
	if(array.empty()) {
		DebugWriteLine(_T("[DataSource::AppendToArrayField] warning:  array is empty; ignoring"));
		return;
	}

	// Update the current state.
	pimpl->Access([name = FromTstring(name), &array, &value](auto& currentState, auto& queue) {
		currentState = UpdateState(currentState, name, [&array](json11::Json state, std::string const& name)->json11::Json {
			if(name.front() == '[') {
				// This is an array field access.
				auto parentArray = state.array_items();
				auto index = std::stoul(name.substr(1));
				if(index >= parentArray.size()) {
					DebugWriteLine(_T("[DataSource::AppendToArrayField] array index %d is out of bounds (%d)"),
						index, parentArray.size());
					throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
				}
				auto& child = parentArray[index];
				if(!child.is_array()) {
					DebugWriteLine(_T("[DataSource::AppendToArrayField] current field value is not an array"));
					throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
				}
				auto childArray = child.array_items();
				childArray.insert(childArray.cend(), array.cbegin(), array.cend());
				child = childArray;
				return parentArray;
			} else {
				auto parentObject = state.object_items();
				auto& child = parentObject[name];
				if(!child.is_array()) {
					DebugWriteLine(_T("[DataSource::AppendToArrayField] current field value is not an array"));
					throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
				}
				auto childArray = child.array_items();
				childArray.insert(childArray.cend(), array.cbegin(), array.cend());
				child = childArray;
				return parentObject;
			}
		}).object_items();

		// Queue the "Append To Array Field" message.
		queue.push_back({ MessageType::Append, name, value });
	});
}
