namespace {
	constexpr size_t maximumConnectionSize = 100'000;
	constexpr size_t maximumDeltaSize = 5'000;
}

struct DataSource::DataSourceImpl {
	using token_t = decltype(Configuration::token);

	struct ConnectMessage {
		token_t token;
		std::vector<tstring> broadcasterIds;
		tstring gameId;
		tstring environment;
		json11::Json data;

		json11::Json to_json() const {
#ifdef _UNICODE
			std::vector<std::string> broadcasterIds_;
			std::transform(broadcasterIds.cbegin(), broadcasterIds.cend(), std::back_inserter(broadcasterIds_), [](auto const& s) {
				return FromTstring(s);
			});
#else
			auto const& broadcasterIds_ = broadcasterIds;
#endif
			json11::Json::object map;
			map.insert({ "token", FromTstring(token) });
			map.insert({ "broadcaster_ids", broadcasterIds_ });
			map.insert({ "game_id", FromTstring(gameId) });
			map.insert({ "env", FromTstring(environment) });
			map.insert({ "data", data });
			return state_t{ { "connect", map } };
		}
	};

	DataSourceImpl(Configuration const& configuration) {
		// Validate the configuration.
		std::string parseError;
		auto json = json11::Json::parse(FromTstring(configuration.initialState), parseError);
		if(!parseError.empty()) {
			DebugWriteLine(_T("[DataSourceImpl::DataSourceImpl] JSON parse error:  %s"), ToTstring(parseError).c_str());
			throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
		}
		if(!json.is_object()) {
			DebugWriteLine(_T("[DataSourceImpl::DataSourceImpl] initial state is not a JSON object"));
			throw TwitchException(FromPlatformError(ERROR_BAD_ARGUMENTS));
		}
		auto state = json.object_items();
		CorrectMetadata(state);
		onTokenExpired = configuration.onTokenExpired;
		token = configuration.token;

		// Set the state.
		synchronizedState.Access([&state](auto& currentState, auto&) {
			currentState = state;
		});

		// Compose the connect function.
		connectFn = [this, broadcasterIds = configuration.broadcasterIds, gameId = configuration.gameId, environment = configuration.environment](state_t const& state) {
			// Send the "Connect" message.
			ConnectMessage connectMessage{
				token,
				broadcasterIds,
				gameId,
				environment,
				state,
			};
			auto const messageText = json11::Json(connectMessage).dump();
			return webSocket.Send(messageText);
		};

		// Compose the open function.
		openFn = [this, timeout = configuration.timeout]() {
			webSocket.OnClosed = [this] { EnqueueReconnect(); };
			webSocket.OnReceived = [this](std::vector<uint8_t>& data) { ProcessMessage(data); };
			return webSocket.Open(url, timeout);
		};

		// Open the WebSocket.
		ThrowIfFailed(openFn());

		// Enqueue a "Reauthorize" message without clearing the token.
		Enqueue({ MessageType::Reauthorize, "", true });

		// Start the sender task.
		auto future = closePromise.get_future().share();
		senderTask = std::async(std::launch::async, [this, future] { SendMessages(future); });
	}

	DataSourceImpl() = delete;
	DataSourceImpl(DataSourceImpl const&) = delete;
	DataSourceImpl(DataSourceImpl&&) = default;
	DataSourceImpl& operator=(DataSourceImpl const&) = delete;
	DataSourceImpl& operator=(DataSourceImpl&&) = default;

	~DataSourceImpl() {
		closePromise.set_value();
		senderTask.get();
	}

	bool GetIsValid() const { return webSocket.IsOpen; }
	__declspec(property(get = GetIsValid)) bool const IsValid;

	template<typename FN>
	void Access(FN fn) {
		synchronizedState.Access(fn);
	}

	void Enqueue(Message const& message) {
		Access([this, &message](auto&, auto& queue) {
#ifdef _DEBUG
			if(!webSocket.IsOpen && message.type != MessageType::Reconnect) {
				DebugWriteLine(_T("[DataSourceImpl::Enqueue] connection not established"));
				assert(webSocket.IsOpen);
			}
#endif
			queue.push_back(message);
		});
	}

private:
	struct Delta {
		std::vector<Message> messages;

		json11::Json to_json() const {
			json11::Json::object rv;
			if(messages.front().type == MessageType::Refresh) {
				json11::Json::object data{ { "data", messages.front().value } };
				rv.insert({ "refresh", data });
			} else {
				json11::Json::array deltas;
				std::copy(messages.cbegin(), messages.cend(), std::back_inserter(deltas));
				rv.insert({ "delta", deltas });
			}
			return rv;
		}
	};

	SynchronizedState synchronizedState;
	WebSocket webSocket;
	std::function<int(state_t const&)> connectFn;
	std::function<int()> openFn;
	std::future<void> senderTask;
	std::promise<void> closePromise;
	token_t token;
	decltype(Configuration::onTokenExpired) onTokenExpired;
	static std::vector<MessageType> const uniqueMessageTypes /*= { MessageType::Reconnect, MessageType::Reauthorize, MessageType::Refresh }*/;

	token_t AcquireToken() {
		std::promise<token_t> promise;
		auto fn = [&promise](token_t const& token) {
			promise.set_value(token);
		};
		try {
			if(onTokenExpired(fn)) {
				// The client indicated it will fetch a token and invoke the call-back.
				return promise.get_future().get();
			} else {
				// The client wants to shut down.
				return token_t();
			}
		} catch(...) {
			// There is likely a logic error in the client.  Shut down the connection.
			DebugWriteLine(_T("[DataSourceImpl::AcquireToken] onTokenExpired threw an exception"));
			return token_t();
		}
	}

	void SendMessages(std::shared_future<void> future) {
		do {
			// Check the queue for unique modifications.
			state_t state;
			Delta delta;
			Access([&state, &delta](auto& currentState, auto& queue) {
				state = currentState;
				for(auto uniqueMessageType : uniqueMessageTypes) {
					auto it = std::find_if(queue.cbegin(), queue.cend(), [uniqueMessageType](auto const& message) { return message.type == uniqueMessageType; });
					if(it != queue.cend()) {
						// Found one; use it.
						delta.messages.push_back(*it);
						queue.clear();
						return;
					}
				}

				// Trim the queue based on repeated modifications.
				if(!queue.empty()) {
					std::remove_reference_t<decltype(queue)> trimmedQueue;
					while(!queue.empty()) {
						auto message = queue.front();
						queue.pop_front();
						switch(message.type) {
						case MessageType::Append:
							ExtendAppend(trimmedQueue, message);
							break;
						case MessageType::Remove:
						case MessageType::Update:
							trimmedQueue.erase(std::remove_if(trimmedQueue.begin(), trimmedQueue.end(), [&message](auto const& trimmedMessage) {
								return trimmedMessage.name == message.name;
							}), trimmedQueue.end());
							trimmedQueue.push_back(message);
							break;
						default:
							assert(false);
						}
					}
					queue.swap(trimmedQueue);
				}

				// Take messages off of the queue until reaching the maximum payload size.
				if(!queue.empty()) {
					while(!queue.empty()) {
						delta.messages.push_back(queue.front());
						if(delta.to_json().dump().size() > maximumDeltaSize) {
							delta.messages.pop_back();
							break;
						}
						queue.pop_front();
					}
					if(delta.messages.empty()) {
						// The first message is larger than the maximum payload size.  Send a "Refresh"
						// message instead.
						queue.clear();
						delta.messages.push_back({ MessageType::Refresh, "", false });
					}
				}
			});

			// If messages were removed from the queue, send them.
			if(!delta.messages.empty()) {
				std::string messageText;
				switch(delta.messages.front().type) {
					int delay, errorCode;
				case MessageType::Reauthorize:
					if(!delta.messages.front().value.bool_value()) {
						token.clear();
					}
					if(!Reauthorize(state)) {
						return;
					}
					break;
				case MessageType::Reconnect:
					// Close the current connection.
					CloseWebSocket();

					// Await the requested delay.
					delay = delta.messages.front().value.int_value() - GetTickCount();
					if(delay > 0) {
						if(future.wait_for(std::chrono::milliseconds(delay)) != std::future_status::timeout) {
							// The client is shutting down.
							return;
						}
					}

					// Open a new connection.
					errorCode = openFn();
					if(errorCode) {
						// Cannot re-establish a connection.
						assert(!webSocket.IsOpen);
						DebugWriteLine(_T("[DataSourceImpl::SendMessages] WebSocket::Open returned error %#x (%d)"),
							errorCode, errorCode & 0xffff);
						return;
					}

					// Perform the "Reauthorize" action above.
					if(!Reauthorize(state)) {
						return;
					}
					break;
				case MessageType::Refresh:
					delta.messages.front().value = state;
					__fallthrough;
				default:
					messageText = delta.to_json().dump();
				}
				if(!messageText.empty()) {
					int errorCode = webSocket.Send(messageText);
					if(errorCode) {
						if(webSocket.IsOpen) {
							// An error occurred.  Enqueue a "Refresh" message and try again.
							DebugWriteLine(_T("[DataSourceImpl::SendMessages] WebSocket::Send returned error %#x (%d)"),
								errorCode, errorCode & 0xffff);
							Access([](auto&, auto& queue) {
								queue.clear();
								queue.push_back({ MessageType::Refresh, "", false });
							});
						} else {
							// Since there is no active connection, either the client wants to shut down or
							// the server either requested a reconnection or unexpectedly closed the
							// connection.  In the first case, the future will be set and this loop will
							// exit.  In the latter cases, the receive or close handler, respectively, will
							// post a "Reconnect" message.  In any case, it's okay to lose the current message.
						}
					}
				}
			}
		} while(future.wait_for(1s) == std::future_status::timeout);

		// Close the connection.
		CloseWebSocket();
	}

	bool Reauthorize(state_t const& state) {
		if(token.empty()) {
			token = AcquireToken();
		}
		if(!token.empty()) {
			auto errorCode = connectFn(state);
			if(!errorCode) {
				return true;
			}
			DebugWriteLine(_T("[DataSourceImpl::Reauthorize] reconnection failed: error %#x (%d)"), errorCode, errorCode & 0xffff);
		}
		CloseWebSocket();
		return false;
	}

	void CloseWebSocket() {
		webSocket.OnClosed = WebSocket::DefaultClosedFn;
		webSocket.OnReceived = WebSocket::DefaultReceivedFn;
		webSocket.Close();
	}

	static void ExtendAppend(std::deque<Message>& queue, Message const& message) {
		auto it = std::find_if(queue.begin(), queue.end(), [&message](auto const& trimmedMessage) {
			return trimmedMessage.type == message.type && trimmedMessage.name == message.name;
		});
		if(it == queue.end()) {
			queue.push_back(message);
		} else {
			assert(it->type == MessageType::Append);
			std::string parseError;
			auto json = it->value;
			assert(parseError.empty());
			assert(json.is_array());
			auto currentArray = json.array_items();
			json = message.value;
			assert(json.is_array());
			auto const& array = json.array_items();
			currentArray.insert(currentArray.cend(), array.cbegin(), array.cend());
			it->value = currentArray;
		}
	}

	void EnqueueReconnect(unsigned reconnectTime = GetTickCount()) {
		// Clear the handlers and enqueue a "Reconnect" message.
		webSocket.OnClosed = WebSocket::DefaultClosedFn;
		webSocket.OnReceived = WebSocket::DefaultReceivedFn;
		Enqueue({ MessageType::Reconnect, "", static_cast<int>(reconnectTime) });
	}

	void ProcessMessage(std::vector<uint8_t>& responseData) {
		responseData.push_back(0); // Add a null terminator.
		std::string parseError;
		auto response = json11::Json::parse(reinterpret_cast<char const*>(responseData.data()), parseError);
		if(!parseError.empty()) {
			DebugWriteLine(_T("[DataSourceImpl::ProcessMessage] invalid JSON from server:  %s"), ToTstring(parseError).c_str());
		}
		auto error = response["error"];
		if(error.is_object()) {
			auto code = error["code"].string_value();
			if(code == "connect_invalid_token") {
				// The server has rejected the authorization token.
				Enqueue({ MessageType::Reauthorize, "", false });
			} else if(code == "connection_not_authed") {
				// Ignore this since it means we're in the middle of getting connected.
			} else {
				DebugWriteLine(_T("[DataSourceImpl::ProcessMessage] error from server:  %s"), ToTstring(code).c_str());
			}
			return;
		}
		auto reconnect = response["reconnect"];
		if(reconnect.is_number()) {
			// The server requested a reconnection.
			auto reconnectTime = reconnect.int_value() + GetTickCount();
			EnqueueReconnect(reconnectTime);
			return;
		}
		if(!response["connected"].bool_value()) {
			DebugWriteLine(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  %s"),
				tstring(responseData.cbegin(), responseData.cend()).c_str());
		}
	}
};

std::vector<MessageType> const DataSource::DataSourceImpl::uniqueMessageTypes = { MessageType::Reconnect, MessageType::Reauthorize, MessageType::Refresh };
