#include "pch.h"
#include "Http.h"
#include "Internal.h"

using namespace Twitch;

namespace {
	tstring const helixFollowRootUrl = _T("https://api.twitch.tv/helix/users/follows");
}

std::future<FollowersResponse> FollowersRequest::Fetch(tstring const& clientId, tstring const& token, tstring const& userId) {
	auto fn = [this, clientId, token, userId] {
		tstring url = helixFollowRootUrl + _T("?to_id=") + userId;
		if(Count != 0) {
			url += _T("&first=") + to_tstring(Count);
		}
		if(Direction != PaginationDirection::None && !Cursor.empty()) {
			url += Direction == PaginationDirection::Forward ? _T("&after=") : _T("&before=");
			url += Cursor;
		}
		auto httpResponse = HttpRequest(clientId, token).Get(url);
		ThrowIfResponseFailed(httpResponse);

		// Parse the response as JSON.
		auto const& response = httpResponse.Response;
		std::string parseError;
		auto jsonVal = json11::Json::parse(std::string(response.cbegin(), response.cend()).c_str(), parseError);
		if(!parseError.empty()) {
			DebugWriteLine(_T("[FollowersRequest::Fetch] JSON parsing failed:  %s"), ToTstring(parseError).c_str());
			throw TwitchException(FromPlatformError(ERROR_INVALID_RESPONSE));
		}

		// Send a science event.
		SendTokenScienceEvent(clientId, token, _T("sdk_follow_get_followers"), { { _T("target_user_id"), userId } });

		// Get references for all of the fields.
		FollowersResponse followResponse;
		followResponse.TotalCount = static_cast<unsigned>(jsonVal["total"].int_value());
		followResponse.Cursor = ToTstring(jsonVal["pagination"]["cursor"].string_value());
		auto const& data = jsonVal["data"];
		for(auto const& follower : data.array_items()) {
			followResponse.Followers.emplace_back(Follower{
				ToTstring(follower["from_id"].string_value()),
				ToTstring(follower["from_name"].string_value()),
				ParseTime(follower["followed_at"].string_value()),
				});
		}
		return followResponse;
	};
	return std::async(std::launch::deferred, fn);
}
