#include "pch.h"
#include "StringUtil.h"
#include "HttpUtil.h"
#include "Http.h"

using namespace Twitch;

namespace {
	string_t const helixAuthorizationPrefix = _T("Bearer ");
	string_t const krakenAuthorizationPrefix = _T("OAuth ");
	string_t const clientIdHeaderName = _T("Client-ID");

	bool IsKrakenUrl(tstring const& url) {
		return url.find(_T("https://api.twitch.tv/kraken/")) == 0;
	}
}

HttpParam::HttpParam(tstring const& name, tstring const& value)
	: paramName(name)
	, paramValue(value) {}

HttpParam::HttpParam(tstring const& name, int value)
	: paramName(name)
	, paramValue(to_tstring(value)) {}

bool Twitch::IsTwitchEndpoint(string_t url) {
	Uri uri(url);
	tstring const host = ToLowerCase(uri.GetHostName());
	return host == _T("twitch.tv") || EndsWith(host, _T(".twitch.tv"));
}

std::vector<char> const HttpRequest::EmptyBody;
tstring const HttpRequest::EmptyString;

HttpRequest::HttpRequest(tstring const& clientId_/*= EmptyString*/, tstring const& token_/*= EmptyString*/) :
	clientId(clientId_),
	token(token_),
	timeout(DefaultTimeout) {}

HttpRequest::~HttpRequest() {}

HttpResponse HttpRequest::Get(tstring const& url) {
	int result;
	uintptr_t handle;
	std::tie(result, handle) = StartHttpRequest(url, HTTP_GET_REQUEST, EmptyBody);
	return HttpResponse(result, handle);
}

HttpResponse HttpRequest::Post(tstring const& url, std::vector<char> const& requestBody) {
	int result;
	uintptr_t handle;
	std::tie(result, handle) = StartHttpRequest(url, HTTP_POST_REQUEST, requestBody);
	return HttpResponse(result, handle);
}

HttpResponse HttpRequest::Put(tstring const& url, std::vector<char> const& requestBody) {
	int result;
	uintptr_t handle;
	std::tie(result, handle) = StartHttpRequest(url, HTTP_PUT_REQUEST, requestBody);
	return HttpResponse(result, handle);
}

std::tuple<int, uintptr_t> HttpRequest::StartHttpRequest(tstring const& url, HttpRequestType requestType, std::vector<char> const& requestBody) {
	// Validate the arguments.
	if(url.empty()) {
		return std::make_tuple(FromPlatformError(ERROR_BAD_ARGUMENTS), 0);
	}
	switch(requestType) {
	case HTTP_GET_REQUEST:
	case HTTP_PUT_REQUEST:
	case HTTP_POST_REQUEST:
	case HTTP_DELETE_REQUEST:
		break;
	default:
		return std::make_tuple(FromPlatformError(ERROR_BAD_ARGUMENTS), 0);
	}

	// Add request headers.
	std::vector<HttpParam> requestHeaders;

	// Add the Kraken Accept header, if necessary.
	bool isKrakenUrl = IsKrakenUrl(url);
	if(isKrakenUrl) {
		requestHeaders.emplace_back(HttpParam(_T("Accept"), _T("application/vnd.twitchtv.v5+json")));
	}

	// Add the authorization token, if provided.
	if(!token.empty()) {
		auto authorizationPrefix = isKrakenUrl ? krakenAuthorizationPrefix : helixAuthorizationPrefix;
		requestHeaders.emplace_back(HttpParam(_T("Authorization"), authorizationPrefix + token));
	}

	// Prefer to receive gzipped responses if no other specified.
	requestHeaders.emplace_back(HttpParam(_T("Accept-Encoding"), _T("gzip")));

	// Add the client ID, if appropriate.
	if(!clientId.empty() && IsTwitchEndpoint(url.c_str())) {
		requestHeaders.emplace_back(HttpParam(clientIdHeaderName, clientId));
	}

	return Platform::StartHttpRequest(url.c_str(), requestType, timeout, requestHeaders, requestBody);
}

HttpResponse::HttpResponse(int result_, uintptr_t handle_) : resultCode(0), result(result_), handle(handle_) {}

HttpResponse::~HttpResponse() {
	Platform::FinishHttpRequest(handle);
}

stringmap const& HttpResponse::GetHeaders() const {
	if(headers.empty()) {
		int result_;
		std::tie(result_, const_cast<HttpResponse*>(this)->headers) = Platform::GetHttpResponseHeaders(handle);
		UNREFERENCED_PARAMETER(result_); // TODO
	}
	return headers;
}

std::vector<char> const& HttpResponse::GetResponse() const {
	if(response.empty()) {
		int result_;
		std::tie(result_, const_cast<HttpResponse*>(this)->response) = Platform::GetHttpResponse(handle);
		UNREFERENCED_PARAMETER(result_); // TODO
	}
	return response;
}

int HttpResponse::GetResultCode() const {
	if(resultCode == 0) {
		const_cast<HttpResponse*>(this)->resultCode = Platform::GetHttpResponseResultCode(handle);
	}
	return resultCode;
}

WebSocket::ClosedFn WebSocket::DefaultClosedFn = [] {};
WebSocket::ReceivedFn WebSocket::DefaultReceivedFn = [](auto) {};
