#include "pch.h"
#include "Internal.h"

using namespace Twitch;

#ifdef _DEBUG
DebugFn test_OutputDebugString;

void Twitch::DebugWriteLine_(string_t format, ...) {
	va_list args;
	va_start(args, format);
# ifdef _WIN32
	size_t const len = _vsctprintf(format, args);
	tstring buffer;
	buffer.resize(len + 1);
	_vstprintf_s(&buffer[0], buffer.size(), format, args);
	buffer.back() = _T('\n');
	if(test_OutputDebugString) {
		test_OutputDebugString(buffer.c_str());
	} else {
		OutputDebugString(buffer.c_str());
	}
# else
	size_t const len = vsnprintf(nullptr, 0, format, args);
	std::string buffer;
	buffer.resize(len + 1);
	vsprintf_s(&buffer[0], buffer.size(), format, args);
	buffer.back() = '\n';
	std::cout << buffer;
# endif
	va_end(args);
}
#endif

tstring Twitch::ExtractToken(tstring const& text, string_t name/*= _T("token")*/) {
	// Extract the GET request line, if any.
	auto token = text;
	auto i = token.find(_T('\r'));
	if(i != tstring::npos) {
		token.erase(i);

		// Extract the URL.
		i = token.find(_T(' '));
		if(i != tstring::npos) {
			token.erase(0, i + 1);
			i = token.find(_T(' '));
			if(i != tstring::npos) {
				token.erase(i);
			}
		}
	}

	// Extract the token.
	tstring s = name;
	s += _T('=');
	i = token.find(s);
	if(i == tstring::npos) {
		return _T("");
	}
	if(i > 0 && _tcschr(_T("#&?"), token[i - 1]) == nullptr) {
		return _T("");
	}
	token.erase(0, i + s.size());
	i = token.find(_T('&'));
	if(i != tstring::npos) {
		token.erase(i);
	}

	return token;
}

static char const base64Alphabet[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

std::string Twitch::Base64Encode(std::vector<char> const& data) {
	std::string rv;
	for(std::vector<char>::size_type i = 0, n = data.size() - data.size() % 3; i < n; i += 3) {
		unsigned index = static_cast<unsigned char>(data[i]) >> 2;
		rv += base64Alphabet[index];
		index = (data[i] << 4) | (static_cast<unsigned char>(data[i + 1]) >> 4);
		rv += base64Alphabet[index & 0x3f];
		index = (data[i + 1] << 2) | (static_cast<unsigned char>(data[i + 2]) >> 6);
		rv += base64Alphabet[index & 0x3f];
		index = data[i + 2];
		rv += base64Alphabet[index & 0x3f];
	}
	switch(data.size() % 3) {
		unsigned index;
	case 1:
		index = static_cast<unsigned char>(data.back()) >> 2;
		rv += base64Alphabet[index];
		index = data.back() << 4;
		rv += base64Alphabet[index & 0x3f];
		rv += "==";
		break;
	case 2:
		index = static_cast<unsigned char>(data[data.size() - 2]) >> 2;
		rv += base64Alphabet[index];
		index = (data[data.size() - 2]) << 4 | (static_cast<unsigned char>(data.back()) >> 4);
		rv += base64Alphabet[index & 0x3f];
		index = static_cast<unsigned char>(data.back()) << 2;
		rv += base64Alphabet[index & 0x3f];
		rv += '=';
		break;
	}
	return rv;
}

std::time_t Twitch::ParseTime(std::string const& s) {
	std::stringstream ss(s);
	std::tm when{};
	auto const* const format = "%Y-%m-%dT%H:%M:%SZ";
	ss >> std::get_time(&when, format);
	return std::mktime(&when);
}

#ifdef _UNICODE
std::string Twitch::FromTstring(tstring const& s) {
# ifdef _WIN32
	std::string rv;
	if(!s.empty()) {
		auto inputSize = static_cast<int>(s.size());
		auto n = WideCharToMultiByte(CP_UTF8, WC_ERR_INVALID_CHARS, &s[0], inputSize, nullptr, 0, nullptr, nullptr);
		if(n == 0) {
			throw TwitchException(FromPlatformError(GetLastError()));
		}
		rv.resize(n, '\0');
		if(WideCharToMultiByte(CP_UTF8, WC_ERR_INVALID_CHARS, &s[0], inputSize, &rv[0], n, nullptr, nullptr) == 0) {
			throw TwitchException(FromPlatformError(GetLastError()));
		}
	}
	return rv;
# else
	std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>> converter;
	return converter.to_bytes(s);
# endif
}
tstring Twitch::ToTstring(std::string const& s) {
# ifdef _WIN32
	std::wstring rv;
	if(!s.empty()) {
		auto inputSize = static_cast<int>(s.size());
		auto n = MultiByteToWideChar(CP_UTF8, MB_ERR_INVALID_CHARS, &s[0], inputSize, nullptr, 0);
		if(n == 0) {
			throw TwitchException(FromPlatformError(GetLastError()));
		}
		rv.resize(n, L'\0');
		if(MultiByteToWideChar(CP_UTF8, MB_ERR_INVALID_CHARS, &s[0], inputSize, &rv[0], n) == 0) {
			throw TwitchException(FromPlatformError(GetLastError()));
		}
	}
	return rv;
# else
	std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>> converter;
	return converter.from_bytes(s);
# endif
}
#endif
