#pragma once

#include "Twitch.h"
#include <json11.hpp>
#include "Http.h"
#include "StringUtil.h"

#ifdef _WIN32
# define ReturnIfError(result) do { if((result) != ERROR_SUCCESS) return tstring(); } while(false)

# ifdef _DEBUG
using DebugFn = void(*)(LPCTSTR);

extern DebugFn test_OutputDebugString;
# endif
#endif

namespace Twitch {
#ifdef _DEBUG
	void DebugWriteLine_(string_t format, ...);
# define DebugWriteLine(...) DebugWriteLine_(__VA_ARGS__)
#else
# define DebugWriteLine(...) ((void)0)
#endif
	tstring ExtractToken(tstring const& s, string_t name = _T("token"));

	template<typename FN>
	auto LockedOperation(std::mutex& m, FN fn) {
		std::lock_guard<decltype(m)> lock(m);
		return fn();
	}

	inline void ThrowIfFailed(int result) {
		if(result < 0) {
			throw TwitchException(result);
		}
	}

	inline void ThrowIfResponseFailed(HttpResponse const& httpResponse) {
		if(httpResponse.Result) {
			throw TwitchException(httpResponse.Result);
		}
		if(!HttpResponse::IsSuccessful(httpResponse.ResultCode)) {
#ifdef _DEBUG
			DebugWriteLine(_T("HttpResponse failure, status %d:\n%s"), httpResponse.ResultCode,
				tstring(httpResponse.Response.cbegin(), httpResponse.Response.cend()).c_str());
#endif
			throw TwitchException(httpResponse.ResultCode);
		}
	}

	template<typename R, typename T>
	struct Handle {
		Handle(T value, std::function<R(T)> destroyFn) : value(value), destroyFn(destroyFn) {}
		~Handle() { destroyFn(value); }
		operator T() const { return value; }

	private:
		T value;
		std::function<R(T)> destroyFn;
	};

#ifdef _WIN32
# if WINAPI_FAMILY_PARTITION(WINAPI_PARTITION_DESKTOP)
	struct RegKey {
		RegKey() : key(HKEY()) {}
		~RegKey() { RegCloseKey(key); }
		HKEY* operator&() { return &key; }
		operator HKEY() const { return key; }

	private:
		HKEY key;
	};
# endif
#endif

	std::string Base64Encode(std::vector<char> const& data);
	tstring GetDeviceId();
	std::time_t ParseTime(std::string const& s);
	void SendTokenScienceEvent(tstring const& clientId, tstring const& token, tstring const& method, stringmap const& properties) noexcept;
	void SendUserScienceEvent(tstring const& clientId, tstring const& userId, tstring const& method, stringmap const& properties) noexcept;

#ifdef _UNICODE
	std::string FromTstring(tstring const& s);
	tstring ToTstring(std::string const& s);
#else
	inline std::string FromTstring(tstring const& s) { return s; }
	inline tstring ToTstring(std::string const& s) { return s; }
# define _istspace isspace
#endif

#ifdef __ORBIS__
	void rand_s(unsigned* p);
#endif
}
