#include "pch.h"
#include "Http.h"
#include "Internal.h"

using namespace Twitch;

namespace {
	tstring const helixStreamsRootUrl = _T("https://api.twitch.tv/helix/streams?");

	std::vector<tstring> CollectIds(json11::Json const& jsonArray) {
		std::vector<tstring> rv;
		if(jsonArray.is_array()) {
			for(auto const& jsonString : jsonArray.array_items()) {
				rv.emplace_back(ToTstring(jsonString.string_value()));
			}
		} else {
			DebugWriteLine(_T("[CollectIds] warning: not an array"));
		}
		return rv;
	}
}

std::future<StreamsResponse> StreamsRequest::Fetch(tstring const& clientId) {
	auto fn = [this, clientId] {
		stringmap properties;
		tstring url = helixStreamsRootUrl;
		if(Count != 0) {
			url += _T("&first=") + to_tstring(Count);
		}
		if(Direction != PaginationDirection::None && !Cursor.empty()) {
			url += Direction == PaginationDirection::Forward ? _T("&after=") : _T("&before=");
			url += Cursor;
		}
		if(!CommunityId.empty()) {
			url += _T("&community_id=") + CommunityId;
			properties.insert({ _T("community_id"), CommunityId });
		}
		if(!GameId.empty()) {
			url += _T("&game_id=") + GameId;
			properties.insert({ _T("game_id"), GameId });
		}
		if(!Language.empty()) {
			url += _T("&language=") + Language;
			properties.insert({ _T("language"), Language });
		}
		if(!UserId.empty()) {
			url += _T("&user_id=") + UserId;
			properties.insert({ _T("user_id"), UserId });
		}
		if(!UserLogin.empty()) {
			auto login = ToLowerCase(UserLogin);
			url += _T("&user_login=") + login;
			properties.insert({ _T("user_login"), login });
		}
		auto httpResponse = HttpRequest(clientId).Get(url);
		ThrowIfResponseFailed(httpResponse);

		// Parse the response as JSON.
		auto const& response = httpResponse.Response;
		std::string parseError;
		auto jsonVal = json11::Json::parse(std::string(response.cbegin(), response.cend()).c_str(), parseError);
		if(!parseError.empty()) {
			DebugWriteLine(_T("[StreamsRequest::Fetch] JSON parsing failed:  %s"), ToTstring(parseError).c_str());
			throw TwitchException(FromPlatformError(ERROR_INVALID_RESPONSE));
		}

		// Send a science event.
		SendUserScienceEvent(clientId, tstring(), _T("sdk_stream_get"), properties);

		// Get references for all of the fields.
		StreamsResponse streamsResponse;
		streamsResponse.Cursor = ToTstring(jsonVal["pagination"]["cursor"].string_value());
		auto const& data = jsonVal["data"];
		for(auto const& stream : data.array_items()) {
			Stream::StreamType type = Stream::StreamType::None;
			if(stream["type"].string_value() == "live") {
				type = Stream::StreamType::Live;
			} else if(stream["type"].string_value() == "vodcast") {
				type = Stream::StreamType::Vodcast;
			}
			streamsResponse.Streams.emplace_back(Stream{
				CollectIds(stream["community_ids"]),
				ToTstring(stream["game_id"].string_value()),
				ToTstring(stream["id"].string_value()),
				ToTstring(stream["language"].string_value()),
				ParseTime(stream["started_at"].string_value()),
				CollectIds(stream["tag_ids"]),
				ToTstring(stream["thumbnail_url"].string_value()),
				ToTstring(stream["title"].string_value()),
				type,
				ToTstring(stream["user_id"].string_value()),
				ToTstring(stream["user_name"].string_value()),
				static_cast<unsigned>(stream["viewer_count"].int_value())
				});
		}
		return streamsResponse;
	};
	return std::async(std::launch::deferred, fn);
}
