// Copyright (c) 2019, Twitch Interactive, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
// OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// OTHER DEALINGS IN THE SOFTWARE.

#pragma once

#ifdef CREATING_TWITCH_SDK_DLL
# define TWITCH_SDK_DLL_API __declspec(dllexport)
#elif defined(USING_TWITCH_SDK_DLL)
# define TWITCH_SDK_DLL_API __declspec(dllimport)
#else
# define TWITCH_SDK_DLL_API
#endif

#ifdef _WIN32
# ifdef _XBOX_ONE
#	 include <d3d11_x.h>
# else
#	 include <d3d11_1.h>
# endif
#endif
#ifdef __ORBIS__
# include <videodec2.h>
#endif
#include <cassert>
#include <ctime>
#include <condition_variable>
#include <exception>
#include <functional>
#include <future>
#include <map>
#include <memory>
#include <mutex>
#include <string>
#include <utility>
#include <vector>

namespace Twitch {
#ifdef _UNICODE
	using string_t = wchar_t const*;
	using tstring = std::wstring;
#else
	using string_t = char const*;
	using tstring = std::string;
#endif
	using stringmap = std::map<tstring, tstring>;

	class TWITCH_SDK_DLL_API TwitchException;
	template<typename T> class Pollable;
	class Callback;
	class TWITCH_SDK_DLL_API BrowserAuth;
	class TWITCH_SDK_DLL_API ServerAuth;
	class TWITCH_SDK_DLL_API ClientAuth;
	class TWITCH_SDK_DLL_API ActivationAuth;
	class TWITCH_SDK_DLL_API ApplicationAuth;
	class TWITCH_SDK_DLL_API User;
	class TWITCH_SDK_DLL_API FollowersRequest;
	class TWITCH_SDK_DLL_API FollowersResponse;
	class TWITCH_SDK_DLL_API Follower;
	class TWITCH_SDK_DLL_API Clip;
	class TWITCH_SDK_DLL_API StreamsRequest;
	class TWITCH_SDK_DLL_API StreamsResponse;
	class TWITCH_SDK_DLL_API Stream;
	class TWITCH_SDK_DLL_API Player;
	class TWITCH_SDK_DLL_API DataSource;
}

#ifdef _WIN32
# pragma warning(disable: 4275) // non dll-interface class 'std::exception' used as base for dll-interface class 'TwitchException'
#endif
class TWITCH_SDK_DLL_API Twitch::TwitchException : public std::exception {
public:
	explicit TwitchException(int errorCode) noexcept;

	char const* what() const noexcept override;
	int GetErrorCode() const noexcept { return errorCode; }

	__declspec(property(get = GetErrorCode)) int const ErrorCode;

private:
	char description[128];
	int errorCode;
};
#ifdef _WIN32
# pragma warning(default: 4275)
#endif

#ifdef _WIN32
# pragma warning(disable: 4251) // class '*' needs to have dll-interface to be used by clients of class 'Twitch::*'
#endif

template<typename T>
class Twitch::Pollable {
public:
	enum class PollableReadyState { NotReady, ValueReady, ExceptionReady };

	explicit Pollable(std::future<T>&& deferredValue) : value(), readyState(PollableReadyState::NotReady) {
		auto fn = [this](std::future<T>&& deferredValue) {
			try {
				value = deferredValue.get();
				readyState = PollableReadyState::ValueReady;
			} catch(...) {
				ex = std::current_exception();
				readyState = PollableReadyState::ExceptionReady;
			}
		};
		task = std::async(std::launch::async, fn, std::move(deferredValue));
	}

	PollableReadyState GetReadyState() const noexcept { return readyState; }
	bool GetIsReady() const noexcept { return readyState != PollableReadyState::NotReady; }
	bool GetHasValue() const noexcept { return readyState == PollableReadyState::ValueReady; }
	bool GetHasException() const noexcept { return readyState == PollableReadyState::ExceptionReady; }
	T const& GetValue() const {
		if(GetHasException()) {
			std::rethrow_exception(ex);
		} else {
			return value;
		}
	}
	std::exception_ptr GetException() const noexcept { return ex; }

	__declspec(property(get = GetReadyState)) PollableReadyState const ReadyState;
	__declspec(property(get = GetIsReady)) bool const IsReady;
	__declspec(property(get = GetHasValue)) bool const HasValue;
	__declspec(property(get = GetHasException)) bool const HasException;
	__declspec(property(get = GetValue)) T const& Value;
	__declspec(property(get = GetException)) std::exception_ptr const Exception;

private:
	std::future<void> task;
	T value;
	std::exception_ptr ex;
	PollableReadyState readyState;
};

class Twitch::Callback {
public:
	Callback() = default;
	Callback(Callback const&) = delete;
	Callback(Callback&&) = default;
	Callback& operator=(Callback const&) = delete;
	Callback& operator=(Callback&&) = default;
	template<typename T, typename S, typename F>
	Callback(std::future<T>&& deferredValue, S onSuccess, F onFailure) {
		auto fn = [onSuccess, onFailure](std::future<T>&& deferredValue) {
			T value;
			try {
				value = deferredValue.get();
			} catch(...) {
				try {
					onFailure(std::current_exception());
				} catch(...) {
					// The onFailure call-back threw an exception.  These are
					// silently ignored in release builds.
					assert(false);
				}
				return;
			}
			try {
				onSuccess(value);
			} catch(...) {
				// The onSuccess call-back threw an exception.  These are
				// silently ignored in release builds.
				assert(false);
			}
		};
		task = std::async(std::launch::async, fn, std::move(deferredValue));
	}
	~Callback() = default;

	void CheckForException() {
		if(task.valid()) {
			task.get();
		}
	}

private:
	std::future<void> task;
};

class TWITCH_SDK_DLL_API Twitch::BrowserAuth {
public:
	BrowserAuth();
	~BrowserAuth();

	template<typename T>
	static tstring ConstructScopes(T const& scopes) {
		tstring scope;
		for(auto s : scopes) {
			scope.append(s);
			scope.append(1, tstring::value_type(' '));
		}
		scope.pop_back();
		return scope;
	}
	std::future<tstring> Launch(tstring const& clientId, tstring const& redirectUri, tstring const& responseType, tstring const& scopes, tstring& state, string_t scheme = nullptr);
	void Stop(tstring const& accessToken);
	tstring GetToken() const;

private:
	std::mutex tokenAccessor;
	using TokenLock = std::unique_lock<decltype(tokenAccessor)>;
#ifndef __ORBIS__
	std::condition_variable cv;
	bool isReady;
#endif
	tstring token;

	std::future<tstring> Launch(tstring const& url, tstring const& state, string_t scheme);
};

class TWITCH_SDK_DLL_API Twitch::ServerAuth : Twitch::BrowserAuth {
public:
	ServerAuth();
	~ServerAuth();

	std::future<tstring> StartAuth(tstring const& clientId, tstring const& redirectUri, tstring const& scopes, tstring& state);
	void Finish(tstring const& accessToken, tstring const& refreshToken);
	void Cancel();
	tstring GetAccessToken() const { return GetToken(); }
	tstring GetRefreshToken() const;

	__declspec(property(get = GetAccessToken)) tstring const AccessToken;
	__declspec(property(get = GetRefreshToken)) tstring const RefreshToken;

private:
	tstring refreshToken;
};

class TWITCH_SDK_DLL_API Twitch::ClientAuth : Twitch::BrowserAuth {
public:
	ClientAuth();
	~ClientAuth();

	std::future<tstring> StartAuth(tstring const& clientId, tstring const& redirectUri, tstring const& scopes);
#ifdef __cplusplus_winrt
	void Finish(tstring activationUri);
#endif
	void Cancel();
	tstring GetAccessToken() const { return GetToken(); }

	__declspec(property(get = GetAccessToken)) tstring const AccessToken;

#ifdef _WIN32
private:
# ifdef __cplusplus_winrt
	tstring state;
# else
	std::shared_ptr<struct DdeServer> ddeServer;
# endif
#endif
};

class TWITCH_SDK_DLL_API Twitch::ActivationAuth {
public:
	static tstring GenerateActivationToken();
	std::future<tstring> StartAuth(tstring const& applicationId, tstring const& activationToken);
	std::future<tstring> AcquireAccessToken(tstring const& applicationId, tstring const& activationToken);
	void Cancel();
	std::future<void> SignOut(tstring const& applicationId, tstring const& activationToken);

private:
	bool isCanceled;
};

class TWITCH_SDK_DLL_API Twitch::ApplicationAuth {
public:
	std::future<tstring> StartAuth(tstring const& clientId, tstring const& clientSecret, tstring const& scopes);
};

class TWITCH_SDK_DLL_API Twitch::User {
public:
	User();
	~User();

	static std::future<User> FetchCurrent(tstring const& clientId, tstring const& token);
	static std::future<User> FetchOtherById(tstring const& clientId, tstring const& id);
	static std::future<User> FetchOtherByLogin(tstring const& clientId, tstring const& login);
	std::future<std::vector<char>> FetchOfflineImage();
	std::future<std::vector<char>> FetchProfileImage();
	std::future<void> UpdateDescription(tstring const& clientId, tstring const& token, tstring const& description);
	static std::future<bool> FollowChannel(tstring const& userId, tstring const& clientId, tstring const& token, tstring const& channelId);
	std::future<bool> FollowChannel(tstring const& clientId, tstring const& token, tstring const& channelId);
	tstring GetBroadcasterType() const { return broadcasterType; }
	tstring GetDescription() const { return description; }
	tstring GetDisplayName() const { return displayName; }
	tstring GetEmail() const { return email; }
	tstring GetId() const { return id; }
	tstring GetLogin() const { return login; }
	tstring GetOfflineImageUrl() const { return offlineImageUrl; }
	tstring GetProfileImageUrl() const { return profileImageUrl; }
	tstring GetType() const { return type; }
	unsigned GetViewCount() const { return viewCount; }

	__declspec(property(get = GetBroadcasterType)) tstring const BroadcasterType;
	__declspec(property(get = GetDescription)) tstring const Description;
	__declspec(property(get = GetDisplayName)) tstring const DisplayName;
	__declspec(property(get = GetEmail)) tstring const Email;
	__declspec(property(get = GetId)) tstring const Id;
	__declspec(property(get = GetLogin)) tstring const Login;
	__declspec(property(get = GetOfflineImageUrl)) tstring const OfflineImageUrl;
	__declspec(property(get = GetProfileImageUrl)) tstring const ProfileImageUrl;
	__declspec(property(get = GetType)) tstring const Type;
	__declspec(property(get = GetViewCount)) unsigned const ViewCount;

private:
	tstring broadcasterType;	// User's broadcaster type: "partner", "affiliate", or "".
	tstring description;		// User's channel description.
	tstring displayName;		// User's display name.
	tstring email;				// User's email address. Returned if the request includes the user:read:edit scope.
	tstring id;					// User's ID.
	tstring login;				// User's login name.
	tstring offlineImageUrl;	// URL of the user's offline image.
	tstring profileImageUrl;	// URL of the user's profile image.
	tstring type;				// User's type: "staff", "admin", "global_mod", or "".
	unsigned viewCount;			// Number of users following this user.

	static std::future<User> FetchOther(tstring const& clientId, string_t const query, tstring const& value);
	static User ParseResponseForUser(std::vector<char> const& response);
};

class TWITCH_SDK_DLL_API Twitch::FollowersRequest {
public:
	enum class PaginationDirection { None, Forward, Backward };

	tstring Cursor;
	PaginationDirection Direction;
	unsigned Count;

	FollowersRequest() : Direction(PaginationDirection::None), Count(0) {}
	FollowersRequest(FollowersRequest const&) = default;
	FollowersRequest(FollowersRequest&&) = default;
	FollowersRequest& operator=(FollowersRequest const&) = default;
	FollowersRequest& operator=(FollowersRequest&&) = default;
	~FollowersRequest() = default;
	std::future<FollowersResponse> Fetch(tstring const& clientId, tstring const& token, tstring const& userId);
};

class TWITCH_SDK_DLL_API Twitch::FollowersResponse {
public:
	std::vector<Follower> Followers;
	tstring Cursor;
	unsigned TotalCount;
};

class TWITCH_SDK_DLL_API Twitch::Follower {
public:
	tstring UserId;
	tstring UserLogin;
	std::time_t FollowedAt;
};

class TWITCH_SDK_DLL_API Twitch::Clip {
public:
	static std::future<tstring> Create(tstring const& clientId, tstring const& token, tstring const& broadcasterId, bool hasDelay = false);
	static std::future<Clip> Fetch(tstring const& clientId, tstring const& id);

	tstring GetId() const { return id; }
	tstring GetUrl() const { return url; }
	tstring GetEmbedUrl() const { return embedUrl; }
	tstring GetBroadcasterId() const { return broadcasterId; }
	tstring GetBroadcasterLogin() const { return broadcasterLogin; }
	tstring GetCreatorId() const { return creatorId; }
	tstring GetCreatorLogin() const { return creatorLogin; }
	tstring GetVideoId() const { return videoId; }
	tstring GetGameId() const { return gameId; }
	tstring GetLanguage() const { return language; }
	tstring GetTitle() const { return title; }
	unsigned GetViewCount() const { return viewCount; }
	tstring GetCreatedAt() const { return createdAt; }
	tstring GetThumbnailUrl() const { return thumbnailUrl; }

	__declspec(property(get = GetId)) tstring const Id;
	__declspec(property(get = GetUrl)) tstring const Url;
	__declspec(property(get = GetEmbedUrl)) tstring const EmbedUrl;
	__declspec(property(get = GetBroadcasterId)) tstring const BroadcasterId;
	__declspec(property(get = GetBroadcasterLogin)) tstring const BroadcasterLogin;
	__declspec(property(get = GetCreatorId)) tstring const CreatorId;
	__declspec(property(get = GetCreatorLogin)) tstring const CreatorLogin;
	__declspec(property(get = GetVideoId)) tstring const VideoId;
	__declspec(property(get = GetGameId)) tstring const GameId;
	__declspec(property(get = GetLanguage)) tstring const Language;
	__declspec(property(get = GetTitle)) tstring const Title;
	__declspec(property(get = GetViewCount)) unsigned const ViewCount;
	__declspec(property(get = GetCreatedAt)) tstring const CreatedAt;
	__declspec(property(get = GetThumbnailUrl)) tstring const ThumbnailUrl;

private:
	tstring broadcasterId;
	tstring broadcasterLogin;
	tstring createdAt;
	tstring creatorId;
	tstring creatorLogin;
	tstring embedUrl;
	tstring gameId;
	tstring id;
	tstring language;
	tstring thumbnailUrl;
	tstring title;
	tstring url;
	tstring videoId;
	unsigned viewCount;
};

class TWITCH_SDK_DLL_API Twitch::StreamsRequest {
public:
	enum class PaginationDirection { None, Forward, Backward };

	tstring Cursor;
	PaginationDirection Direction;
	unsigned Count;
	tstring CommunityId;
	tstring GameId;
	tstring Language;
	tstring UserId;
	tstring UserLogin;

	StreamsRequest() : Direction(PaginationDirection::None), Count(0) {}
	StreamsRequest(StreamsRequest const&) = default;
	StreamsRequest(StreamsRequest&&) = default;
	StreamsRequest& operator=(StreamsRequest const&) = default;
	StreamsRequest& operator=(StreamsRequest&&) = default;
	~StreamsRequest() = default;
	std::future<StreamsResponse> Fetch(tstring const& clientId);
};

class TWITCH_SDK_DLL_API Twitch::StreamsResponse {
public:
	std::vector<Stream> Streams;
	tstring Cursor;
};

class TWITCH_SDK_DLL_API Twitch::Stream {
public:
	enum class StreamType { None, Live, Vodcast };

	std::vector<tstring> CommunityIds;	// Collection of community IDs.
	tstring GameId;						// ID of the game being played on the stream.
	tstring Id;							// Stream ID.
	tstring Language;					// Stream language.
	std::time_t StartedAt;				// UTC timestamp.
	std::vector<tstring> TagIds;		// Collection of tag IDs.
	tstring ThumbnailUrl;				// Thumbnail URL of the stream.  All image URLs have variable width and height.  You can replace {width} and {height} with any values to get that size image.
	tstring Title;						// Stream title.
	StreamType Type;					// Stream type: "live", "vodcast", or "".
	tstring UserId;						// ID of the user who is streaming.
	tstring UserLogin;					// Login of the user who is streaming.
	unsigned ViewerCount;				// Number of viewers watching the stream at the time of the query.
};

class TWITCH_SDK_DLL_API Twitch::Player {
public:
	Player() = delete;
	Player(Player const&) = delete;
	Player(Player&&) noexcept;
	Player& operator=(Player const&) = delete;
	Player& operator=(Player&&) noexcept;
	~Player();

	enum class PlayerState { Idle, Ready, Buffering, Playing, Ended };

	class Observer {
	public:
		Observer() = default;
		virtual ~Observer() = default;
		virtual void OnDurationChanged(double duration) = 0;
		virtual void OnError(tstring const& message) = 0;
		virtual void OnPositionChanged(double position) = 0;
		virtual void OnRebuffering() = 0;
		virtual void OnSeekCompleted(double time) = 0;
		virtual void OnStateChanged(PlayerState state) = 0;
	};

#ifdef _WIN32
	using RenderFn = std::function<void(ID3D11Texture2D* texture, int width, int height)>;

	Player(Observer& observer, ID3D11Device1* device, RenderFn onFrame);
#endif
#ifdef __ORBIS__
	using AccessFn = std::function<bool(SceVideodec2OutputInfo const&, SceVideodec2AvcPictureInfo const&)>;

	Player(Observer& observer);

	bool Access(AccessFn fn);
#endif

	void Load(tstring const& url);
	void Play();
	void Pause();

	bool GetIsSeekable() const;
	double GetDuration() const;
	double GetPosition() const;
	void SetPosition(double position);
	double GetBufferedPosition() const;
	PlayerState GetState() const;
	float GetPlaybackRate() const;
	void SetPlaybackRate(float playbackRate);
	bool GetIsLooping() const;
	void SetIsLooping(bool isLooping);
	bool GetIsMuted() const;
	void SetIsMuted(bool isMuted);
	float GetVolume() const;
	void SetVolume(float volume);

	__declspec(property(get = GetIsSeekable)) bool const IsSeekable;
	__declspec(property(get = GetDuration)) double const Duration;
	__declspec(property(get = GetPosition, put = SetPosition)) double Position;
	__declspec(property(get = GetBufferedPosition)) double const BufferedPosition;
	__declspec(property(get = GetState)) PlayerState const State;
	__declspec(property(get = GetPlaybackRate, put = SetPlaybackRate)) float PlaybackRate;
	__declspec(property(get = GetIsLooping, put = SetIsLooping)) bool IsLooping;
	__declspec(property(get = GetIsMuted, put = SetIsMuted)) bool IsMuted;
	__declspec(property(get = GetVolume, put = SetVolume)) float Volume;

private:
	struct PlayerImpl;

	std::unique_ptr<PlayerImpl> pimpl;
};

class TWITCH_SDK_DLL_API Twitch::DataSource {
public:
	using TokenRefreshFn = std::function<void(tstring const&)>;
	using TokenExpiredFn = std::function<bool(TokenRefreshFn)>;

	struct Configuration {
		Configuration() = default;
		Configuration(Configuration const&) = default;
		Configuration(Configuration&&) = default;
		Configuration& operator=(Configuration const&) = default;
		Configuration& operator=(Configuration&&) = default;
		~Configuration() = default;

		tstring token;
		std::vector<tstring> broadcasterIds;
		tstring gameId;
		tstring environment;
		TokenExpiredFn onTokenExpired;
		tstring initialState = EmptyState;
		std::chrono::milliseconds timeout = DefaultTimeout;

		static constexpr std::chrono::seconds DefaultTimeout = std::chrono::seconds(10);
	};

	DataSource();
	DataSource(DataSource const&) = delete;
	DataSource(DataSource&&) noexcept;
	DataSource& operator=(DataSource const&) = delete;
	DataSource& operator=(DataSource&&) noexcept;
	~DataSource();

	void Connect(Configuration const& configuration);
	void Disconnect();
	void ReplaceState(tstring const& state);
	void RemoveField(tstring const& name);
	void UpdateField(tstring const& name, tstring const& value);
	void AppendToArrayField(tstring const& name, tstring const& value);

	static tstring const EmptyState /*= _T("{}")*/;

private:
	struct DataSourceImpl;

	std::unique_ptr<DataSourceImpl> pimpl;
};

#ifdef _WIN32
# pragma warning(default: 4251)
#endif
