#include "pch.h"
#include "Http.h"
#include "HttpUtil.h"
#include "Internal.h"

using namespace Twitch;

namespace {
	tstring const defaultProfileImageUrl = _T("https://static-cdn.jtvnw.net/jtv_user_pictures/xarth/404_user_300x300.png");
	tstring const helixUsersRootUrl = _T("https://api.twitch.tv/helix/users");
}

User::User() : viewCount(0) {}

User::~User() {}

User User::ParseResponseForUser(std::vector<char> const& response) {
	// Parse the response as JSON.
	std::string parseError;
	auto jsonVal = json11::Json::parse(std::string(response.cbegin(), response.cend()).c_str(), parseError);
	if(!parseError.empty()) {
		DebugWriteLine(_T("[User::ParseResponseForUser] JSON parsing failed:  %s"), ToTstring(parseError).c_str());
		throw TwitchException(FromPlatformError(ERROR_INVALID_RESPONSE));
	}

	// Get references for all of the user fields.
	auto const& jUserInfo = jsonVal["data"][0];
	auto const& jBroadcasterType = jUserInfo["broadcaster_type"];
	auto const& jDescription = jUserInfo["description"];
	auto const& jDisplayName = jUserInfo["display_name"];
	auto const& jEmail = jUserInfo["email"];
	auto const& jId = jUserInfo["id"];
	auto const& jLogin = jUserInfo["login"];
	auto const& jOfflineImageUrl = jUserInfo["offline_image_url"];
	auto const& jProfileImageUrl = jUserInfo["profile_image_url"];
	auto const& jType = jUserInfo["type"];
	auto const& jViewCount = jUserInfo["view_count"];

	// Populate and return a User instance.
	User user;
	user.login = ToTstring(jLogin.string_value());
	user.displayName = ToTstring(jDisplayName.string_value());
	user.broadcasterType = ToTstring(jBroadcasterType.string_value());
	user.description = ToTstring(jDescription.string_value());
	user.email = ToTstring(jEmail.string_value());
	user.offlineImageUrl = ToTstring(jOfflineImageUrl.string_value());
	user.profileImageUrl = ToTstring(jProfileImageUrl.string_value());
	user.type = ToTstring(jType.string_value());
	user.viewCount = static_cast<unsigned>(jViewCount.int_value());
	user.id = ToTstring(jId.string_value());
	return user;
}

std::future<User> User::FetchCurrent(tstring const& clientId, tstring const& token) {
	auto fn = [token, clientId]() {
		// Fetch and parse the Helix user.
		auto httpResponse = HttpRequest(clientId, token).Get(helixUsersRootUrl);
		ThrowIfResponseFailed(httpResponse);
		auto user = ParseResponseForUser(httpResponse.Response);

		// Send a science event.
		SendUserScienceEvent(clientId, user.Id, _T("sdk_user_get"), {});

		return user;
	};
	return std::async(std::launch::deferred, fn);
}

std::future<User> User::FetchOtherByLogin(tstring const& clientId, tstring const& login) {
	return FetchOther(clientId, _T("?login="), login);
}

std::future<User> User::FetchOtherById(tstring const& clientId, tstring const& id) {
	return FetchOther(clientId, _T("?id="), id);
}

std::future<User> User::FetchOther(tstring const& clientId, string_t const query, tstring const& value) {
	auto fn = [clientId, query = tstring(query), value]() {
		// Fetch and parse the Helix user.
		tstring url = helixUsersRootUrl + query + value;
		auto httpResponse = HttpRequest(clientId).Get(url);
		ThrowIfResponseFailed(httpResponse);
		auto user = ParseResponseForUser(httpResponse.Response);

		// Send a science event.
		auto key = query[2] == _T('l') ? _T("login") : _T("user_id");
		SendUserScienceEvent(clientId, tstring(), _T("sdk_user_get"), { { key, value } });

		return user;
	};
	return std::async(std::launch::deferred, fn);
}

static std::future<std::vector<char>> FetchImage(tstring const& imageUrl) {
	auto fn = [imageUrl]() {
		auto const& url = imageUrl.empty() ? defaultProfileImageUrl : imageUrl;
		auto httpResponse = HttpRequest().Get(url);
		ThrowIfResponseFailed(httpResponse);
		return httpResponse.Response;
	};
	return std::async(std::launch::deferred, fn);
}

std::future<std::vector<char>> User::FetchOfflineImage() {
	return FetchImage(offlineImageUrl);
}

std::future<std::vector<char>> User::FetchProfileImage() {
	return FetchImage(profileImageUrl);
}

std::future<void> User::UpdateDescription(tstring const& clientId, tstring const& token, tstring const& newDescription) {
	auto fn = [clientId, token, userId = Id, newDescription]() {
		tstring url = helixUsersRootUrl + _T("?description=") + UrlEncode(newDescription);
		auto httpResponse = HttpRequest(clientId, token).Put(url);
		ThrowIfResponseFailed(httpResponse);

		// Send a science event.
		auto descriptionSize = to_tstring(newDescription.size());
		SendUserScienceEvent(clientId, userId, _T("sdk_user_update"), { { _T("description_size"), descriptionSize } });
	};
	return std::async(std::launch::deferred, fn);
}

std::future<bool> Twitch::User::FollowChannel(tstring const& clientId, tstring const& token, tstring const& channelId) {
	return FollowChannel(Id, clientId, token, channelId);
}

std::future<bool> Twitch::User::FollowChannel(tstring const& userId, tstring const& clientId, tstring const& token, tstring const& channelId) {
	auto fn = [clientId, token, userId, channelId]() {
		tstring url = _T("https://api.twitch.tv/kraken/users/") + userId + _T("/follows/channels/") + channelId;
		auto httpResponse = HttpRequest(clientId, token).Put(url);
		if(httpResponse.Result) {
			throw TwitchException(httpResponse.Result);
		}

		// Send a science event.
		SendUserScienceEvent(clientId, userId, _T("sdk_user_update"), { { _T("follow"), channelId } });

		if(httpResponse.ResultCode == 200) {
			return true;
		}
#ifdef _DEBUG
		auto body = httpResponse.Response;
		std::string parseError;
		auto jsonVal = json11::Json::parse(std::string(body.cbegin(), body.cend()).c_str(), parseError);
		if(!parseError.empty()) {
			DebugWriteLine(_T("[User::FollowChannel] status code %d"), httpResponse.ResultCode);
			DebugWriteLine(tstring(body.cbegin(), body.cend()).c_str());
		} else {
			DebugWriteLine(_T("[User::FollowChannel] JSON parsing failed:  %s"), ToTstring(parseError).c_str());
		}
#endif
		return false;
	};
	return std::async(std::launch::deferred, fn);
}
