#include "stdafx.h"
#include "nativeplayer.hpp"
#include "DebugRenderer.h"
#include "DirectXHelper.h"

using namespace Twitch;

namespace {
	constexpr string_t playerStates[] = { _T("Idle"), _T("Ready"), _T("Buffering"), _T("Playing"), _T("Ended") };

	std::string string_format(const char* fmt, ...) {
		int size = (static_cast<int>(strlen(fmt))) * 2 + 50;   // Use a rubric appropriate for your code
		std::string str;
		va_list ap;

		while(1) {     // Maximum two passes on a POSIX system...
			str.resize(size);
			va_start(ap, fmt);
			int n = vsnprintf((char *)str.data(), size, fmt, ap);
			va_end(ap);

			if(n > -1 && n < size) {  // Everything worked
				str.resize(n);
				return str;
			}

			if(n > -1) { // Needed size returned
				size = n + 1;    // For null char
			} else {
				size *= 2;    // Guess at a larger size (OS specific)
			}
		}

		return str;
	}
}

using namespace nativeplayer;
using namespace Microsoft::WRL;

// Initializes D2D resources used for text rendering.
DebugRenderer::DebugRenderer(const std::shared_ptr<DX::DeviceResources>& deviceResources, Application& application)
	: Renderer(deviceResources)
	, application(application)
	, textMetrics() {
	// Create device independent resources
	ComPtr<IDWriteTextFormat> textFormat_;
	DX::ThrowIfFailed(
		deviceResources->GetDWriteFactory()->CreateTextFormat(
			L"Consolas",
			nullptr,
			DWRITE_FONT_WEIGHT_LIGHT,
			DWRITE_FONT_STYLE_NORMAL,
			DWRITE_FONT_STRETCH_NORMAL,
			16.0f,
			L"en-US",
			&textFormat_
		)
	);

	DX::ThrowIfFailed(
		textFormat_.As(&textFormat)
	);

	DX::ThrowIfFailed(
		textFormat->SetParagraphAlignment(DWRITE_PARAGRAPH_ALIGNMENT_NEAR)
	);

	DX::ThrowIfFailed(
		deviceResources->GetD2DFactory()->CreateDrawingStateBlock(&stateBlock)
	);

	CreateDeviceDependentResources();
}

// Updates the text to be displayed.
void DebugRenderer::Update(DX::StepTimer const& /*timer*/) {
	auto mediaPlayer = application.getMediaPlayer();

	bool canSeek = mediaPlayer->IsSeekable;
	auto position = mediaPlayer->Position;
	auto duration = mediaPlayer->Duration;

	using namespace std::chrono;
	bool isLive = duration == std::numeric_limits<double>::infinity();

	auto state = playerStates[static_cast<int>(mediaPlayer->State)];

	tstringstream formattedText;
	formattedText << _T("State: ") << state << std::endl;
	formattedText << _T("CanSeek: ") << (canSeek ? _T("true") : _T("false")) << std::endl;
	formattedText << _T("Position : ") << position << std::endl;

	if(isLive) {
		formattedText << _T("Duration: <LIVE>");
	} else {
		formattedText << _T("Duration: ") << duration;
	}

#ifdef _UNICODE
	text = formattedText.str();
#else
	auto s = formattedText.str();
	text = std::wstring(s.cbegin(), s.cend());
#endif

	ComPtr<IDWriteTextLayout> textLayout_;
	DX::ThrowIfFailed(
		deviceResources->GetDWriteFactory()->CreateTextLayout(
			text.c_str(),
			(uint32)text.length(),
			textFormat.Get(),
			640.0f, // Max width of the input text.
			50.0f, // Max height of the input text.
			&textLayout_
		)
	);

	DX::ThrowIfFailed(
		textLayout_.As(&textLayout)
	);

	DX::ThrowIfFailed(
		textLayout->GetMetrics(&textMetrics)
	);
}

// Renders a frame to the screen.
void DebugRenderer::Render() {
	ID2D1DeviceContext *context = deviceResources->GetD2DDeviceContext();
	Windows::Foundation::Size logicalSize = deviceResources->GetLogicalSize();

	context->SaveDrawingState(stateBlock.Get());
	context->BeginDraw();

	// Position on the bottom right corner
	D2D1::Matrix3x2F screenTranslation = D2D1::Matrix3x2F::Translation(
		logicalSize.Width - textMetrics.layoutWidth,
		logicalSize.Height - textMetrics.height
	);

	context->SetTransform(screenTranslation * deviceResources->GetOrientationTransform2D());

	DX::ThrowIfFailed(
		textFormat->SetTextAlignment(DWRITE_TEXT_ALIGNMENT_TRAILING)
	);

	context->DrawTextLayout(D2D1::Point2F(1.0f, 1.f), textLayout.Get(), background.Get());
	context->DrawTextLayout(D2D1::Point2F(0.f, 0.f), textLayout.Get(), whiteBrush.Get());

	// Ignore D2DERR_RECREATE_TARGET here. This error indicates that the device
	// is lost. It will be handled during the next call to Present.
	HRESULT hr = context->EndDraw();

	if(hr != D2DERR_RECREATE_TARGET) {
		DX::ThrowIfFailed(hr);
	}

	context->RestoreDrawingState(stateBlock.Get());
}

void DebugRenderer::CreateDeviceDependentResources() {
	DX::ThrowIfFailed(deviceResources->GetD2DDeviceContext()->CreateSolidColorBrush(D2D1::ColorF(D2D1::ColorF::White), &whiteBrush));
	DX::ThrowIfFailed(deviceResources->GetD2DDeviceContext()->CreateSolidColorBrush(D2D1::ColorF(D2D1::ColorF::Black, 0.75f), &background));
}

void DebugRenderer::ReleaseDeviceDependentResources() {
	whiteBrush.Reset();
	background.Reset();
}
