#include "stdafx.h"
#include "nativeplayer.hpp"
#include "TimelineRenderer.h"
#include "DirectXHelper.h"

using namespace nativeplayer;
using namespace Microsoft::WRL;

// Initializes D2D resources used for text rendering.
TimelineRenderer::TimelineRenderer(std::shared_ptr<DX::DeviceResources> const& deviceResources, Application& application)
	: Renderer(deviceResources)
	, application(application)

{
	// Create device independent resources
	ComPtr<IDWriteTextFormat> textFormat_;
	DX::ThrowIfFailed(
		deviceResources->GetDWriteFactory()->CreateTextFormat(
			L"Consolas",
			nullptr,
			DWRITE_FONT_WEIGHT_LIGHT,
			DWRITE_FONT_STYLE_NORMAL,
			DWRITE_FONT_STRETCH_NORMAL,
			12.0f,
			L"en-US",
			&textFormat_
		)
	);

	DX::ThrowIfFailed(
		textFormat_.As(&textFormat)
	);

	DX::ThrowIfFailed(
		textFormat->SetParagraphAlignment(DWRITE_PARAGRAPH_ALIGNMENT_CENTER)
	);

	DX::ThrowIfFailed(
		deviceResources->GetD2DFactory()->CreateDrawingStateBlock(&stateBlock)
	);

	CreateDeviceDependentResources();
}

// Updates the text to be displayed.
void TimelineRenderer::Update(DX::StepTimer const & /*timer*/) {
	Windows::Foundation::Size logicalSize = deviceResources->GetLogicalSize();

	// Figure out bounds of the widget
	bounds.left = logicalSize.Width * 0.1f;
	bounds.top = logicalSize.Height - 100.f;
	bounds.right = logicalSize.Width * 0.9f;
	bounds.bottom = bounds.top + 10.0f;

	// some generic offset to move the time-line to the left to avoid going over the debug display
	const int xOffset = -50;
	bounds.left += xOffset;
	bounds.right += xOffset;

	// prevent cropping on the left
	if(bounds.left < 0) {
		bounds.right += 0 - bounds.left;
		bounds.left += 0 - bounds.left;
	}

	// End bounds

	auto mediaPlayer = application.getMediaPlayer();
	double durationInSeconds = mediaPlayer->IsSeekable ? mediaPlayer->Duration : 1.0;

	for(unsigned int index = 0; index < TIMELINE_STEP_COUNT; ++index) {

		double seconds = static_cast<double>(index) / static_cast<double>(TIMELINE_STEP_COUNT - 1) * durationInSeconds;
		char data[16];
		std::snprintf(data, sizeof(data), "|\n%02.2f", seconds);
		text[index] = stringtowstring(data);

		ComPtr<IDWriteTextLayout> textLayout_;
		DX::ThrowIfFailed(deviceResources->GetDWriteFactory()->CreateTextLayout(text[index].c_str(), (uint32)text[index].length(), textFormat.Get(), TIME_STEP_WIDTH, TIME_STEP_HEIGHT, &textLayout_));
		DX::ThrowIfFailed(textLayout_.As(&textLayout[index]));
	}
}

// Renders a frame to the screen.
void TimelineRenderer::Render() {
	auto mediaPlayer = application.getMediaPlayer();
	if(!mediaPlayer->IsSeekable) {
		return;
	}

	ID2D1DeviceContext *context = deviceResources->GetD2DDeviceContext();
	context->SaveDrawingState(stateBlock.Get());
	context->BeginDraw();

	context->FillRectangle(bounds, alphaBrush.Get());

	auto position = mediaPlayer->Position;
	auto duration = mediaPlayer->Duration;

	double ratio = duration == std::numeric_limits<double>::infinity() ? 1.0 : position / duration;

	// Draw the current ratio using a circle
	D2D1_ELLIPSE ellipse;
	ellipse.point.y = (bounds.bottom + bounds.top) / 2;
	ellipse.point.x = (bounds.right - bounds.left) * static_cast<float>(ratio) + bounds.left;
	ellipse.radiusX = 7.5f;
	ellipse.radiusY = 10.0f;
	context->FillEllipse(ellipse, yellowBrush.Get());

	DX::ThrowIfFailed(textFormat->SetTextAlignment(DWRITE_TEXT_ALIGNMENT_CENTER));

	const float stepWidth = (bounds.right - bounds.left) / static_cast<float>(TIMELINE_STEP_COUNT - 1);
	for(unsigned int index = 0; index < TIMELINE_STEP_COUNT; ++index) {
		float offsetX = stepWidth * static_cast<float>(index) - TIME_STEP_WIDTH * 0.5f;
		context->DrawTextLayout(D2D1::Point2F(offsetX + bounds.left + 1.0f, bounds.top + 1.f), textLayout[index].Get(), background.Get());
		context->DrawTextLayout(D2D1::Point2F(offsetX + bounds.left, bounds.top), textLayout[index].Get(), whiteBrush.Get());
	}

	// Ignore D2DERR_RECREATE_TARGET here. This error indicates that the device
	// is lost. It will be handled during the next call to Present.
	HRESULT hr = context->EndDraw();

	if(hr != D2DERR_RECREATE_TARGET) {
		DX::ThrowIfFailed(hr);
	}

	context->RestoreDrawingState(stateBlock.Get());
}

void TimelineRenderer::CreateDeviceDependentResources() {
	DX::ThrowIfFailed(deviceResources->GetD2DDeviceContext()->CreateSolidColorBrush(D2D1::ColorF(D2D1::ColorF::White, 0.5f), &alphaBrush));
	DX::ThrowIfFailed(deviceResources->GetD2DDeviceContext()->CreateSolidColorBrush(D2D1::ColorF(D2D1::ColorF::White), &whiteBrush));
	DX::ThrowIfFailed(deviceResources->GetD2DDeviceContext()->CreateSolidColorBrush(D2D1::ColorF(D2D1::ColorF::Yellow, 0.75f), &yellowBrush));
	DX::ThrowIfFailed(deviceResources->GetD2DDeviceContext()->CreateSolidColorBrush(D2D1::ColorF(D2D1::ColorF::Black, 0.75f), &background));
}

void TimelineRenderer::ReleaseDeviceDependentResources() {
	alphaBrush.Reset();
	whiteBrush.Reset();
	yellowBrush.Reset();
	background.Reset();
}

bool TimelineRenderer::IsInWindow(Windows::Foundation::Point point) {
	return (point.x >= bounds.left && point.x < bounds.right && point.y >= bounds.top && point.y < bounds.bottom);
}

void TimelineRenderer::OnPointerPressed(Windows::Foundation::Point) {}

void TimelineRenderer::OnPointerReleased(Windows::Foundation::Point point) {
	float ratio = (point.x - bounds.left) / static_cast<float>(bounds.right - bounds.left);
	ratio = std::clamp(ratio, 0.f, 1.f);
	auto mediaPlayer = application.getMediaPlayer();
	auto duration = mediaPlayer->Duration;
	auto seekTo = duration * ratio;
	mediaPlayer->Position = seekTo;
}
