#include "stdafx.h"
#include "nativeplayer.hpp"

using namespace Twitch;

namespace {
	constexpr string_t streamUrls[] = {
		_T("https://www.twitch.tv/monstercat"),
		_T("https://www.twitch.tv/videos/v205550776"),
		_T("https://www.twitch.tv/videos/93348507"),
		_T("https://www.twitch.tv/videos/133128389"),
		_T("https://www.twitch.tv/videos/90812612"),
		_T("https://www.twitch.tv/videos/4878428"),
		_T("https://www.twitch.tv/videos/258615219"),
		_T("https://twitch.tv/videos/402798647"),
	};
}

void Application::OnStateChanged(PlayerState newState) {
	m_playerState = newState;
}

void Application::OnError(tstring const& message) {
	lastError = message;
}

int Application::Initialize(Microsoft::WRL::ComPtr<ID3D11Device1> device) {
	auto onRenderedFrame = [this](ID3D11Texture2D* texture, int width, int height) {
		if(OnRenderCallback) {
			OnRenderCallback(texture, width, height);
		}
	};
	m_player = std::make_shared<Player>(*this, device.Get(), onRenderedFrame);
	auto url = streamUrls[0];
	m_player->Load(url);
	return 0;
}

void Application::HandleStartSeek() {
	if(m_playerState == PlayerState::Ready && !m_playerStarted) {
		m_player->Play();
		m_playerStarted = true;
	}
}

int Application::Update() {
	HandleStartSeek();
	return lastError.empty() ? 0 : -1;
}

void Application::Render() {
	// Render performs in the display thread.
	std::this_thread::sleep_for(std::chrono::milliseconds(16));
}

int Application::Finalize() {
	m_player->Pause();
	m_player.reset();
	assert(m_player.use_count() == 0);
	return 0;
}

void Application::SelectNextVideo() {
	m_currentVideoIndex = (m_currentVideoIndex + 1) % _countof(streamUrls);
	auto url = streamUrls[m_currentVideoIndex];
	m_player->Load(url);
}

void Application::SelectPreviousVideo() {
	if(--m_currentVideoIndex < 0) {
		m_currentVideoIndex = static_cast<int>(_countof(streamUrls) - 1);
	}
	auto url = streamUrls[m_currentVideoIndex];
	m_player->Load(url);
}

void Application::Play() {
	m_player->Play();
}

void Application::Pause() {
	m_player->Pause();
}

void Application::SeekForward5Seconds() {
	double newPosition = std::min(m_player->Position + 5.0, m_player->Duration);
	m_player->Position = newPosition;
}

void Application::SeekForward30Seconds() {
	double newPosition = std::min(m_player->Position + 30.0, m_player->Duration);
	m_player->Position = newPosition;
}

void Application::SeekBackward5Seconds() {
	double newPosition = std::max(0.0, m_player->Position - 5.0);
	m_player->Position = newPosition;
}

void Application::SeekBackward30Seconds() {
	double newPosition = std::max(0.0, m_player->Position - 30.0);
	m_player->Position = newPosition;
}

