#include "stdafx.h"
#include "nativeplayer.hpp"
#include "nativeplayerMain.h"
#include "DirectXHelper.h"
#include "ControlRenderer.h"
#include "DebugRenderer.h"
#include "TimelineRenderer.h"

using namespace nativeplayer;
using namespace Concurrency;

// Loads and initializes application assets when the application is loaded.
nativeplayerMain::nativeplayerMain(std::shared_ptr<DX::DeviceResources> const& deviceResources, Application& application) :
    deviceResources(deviceResources)
{
    // Register to be notified if the Device is lost or recreated
    deviceResources->RegisterDeviceNotify(this);

    m_sceneRenderer = std::make_unique<Sample3DSceneRenderer>(deviceResources, application);
    m_renderers.push_back(std::make_shared<DebugRenderer>(deviceResources, application));
    m_renderers.push_back(std::make_shared<TimelineRenderer>(deviceResources, application));
    m_renderers.push_back(std::make_shared<ControlRenderer>(deviceResources, "[ <<| ]\n   Y", std::bind(&Application::SelectPreviousVideo, &application)));
    m_renderers.push_back(std::make_shared<ControlRenderer>(deviceResources, "[ <<-30 ]\n    Up", std::bind(&Application::SeekBackward30Seconds, &application)));
    m_renderers.push_back(std::make_shared<ControlRenderer>(deviceResources, "[ <<-5 ]\n  Left", std::bind(&Application::SeekBackward5Seconds, &application)));
    m_renderers.push_back(std::make_shared<ControlRenderer>(deviceResources, "[ Play ]\n   A", std::bind(&Application::Play, &application)));
    m_renderers.push_back(std::make_shared<ControlRenderer>(deviceResources, "[ Pause ]\n    X", std::bind(&Application::Pause, &application)));
    m_renderers.push_back(std::make_shared<ControlRenderer>(deviceResources, "[ >>+5 ]\n  Right", std::bind(&Application::SeekForward5Seconds, &application)));
    m_renderers.push_back(std::make_shared<ControlRenderer>(deviceResources, "[ >>+30 ]\n   Down", std::bind(&Application::SeekForward30Seconds, &application)));
    m_renderers.push_back(std::make_shared<ControlRenderer>(deviceResources, "[ >>| ]\n   B", std::bind(&Application::SelectNextVideo, &application)));
}

nativeplayerMain::~nativeplayerMain()
{
    // De-register device notification
    deviceResources->RegisterDeviceNotify(nullptr);
}

// Updates application state when the window size changes (e.g. device orientation change)
void nativeplayerMain::CreateWindowSizeDependentResources()
{
    m_sceneRenderer->CreateWindowSizeDependentResources();
}

// Updates the application state once per frame.
void nativeplayerMain::Update()
{
    ProcessInput();

    // Update scene objects.
    m_timer.Tick([&]() {
        m_sceneRenderer->Update(m_timer);

        for (auto & renderer : m_renderers) {
            renderer->Update(m_timer);
        }
    });
}

void nativeplayerMain::OnPointerPressed(POINT point)
{
    for (auto & renderer : m_renderers) {
        if (renderer->IsInWindow(point)) {
            renderer->OnPointerPressed(point);
            break;
        }
    }
}

void nativeplayerMain::OnPointerReleased(POINT point)
{
    for (auto & renderer : m_renderers) {
        if (renderer->IsInWindow(point)) {
            renderer->OnPointerReleased(point);
            break;
        }
    }
}

// Process all input from the user before updating game state
void nativeplayerMain::ProcessInput()
{
    // TODO: Add per frame input handling here.
    m_sceneRenderer->TrackingUpdate(m_pointerLocation);
}


// Renders the current frame according to the current application state.
// Returns true if the frame was rendered and is ready to be displayed.
bool nativeplayerMain::Render()
{
    // Don't try to render anything before the first Update.
    if (m_timer.GetFrameCount() == 0) {
        return false;
    }

    auto context = deviceResources->GetD3DDeviceContext();

    // Reset the viewport to target the whole screen.
    auto viewport = deviceResources->GetScreenViewport();
    context->RSSetViewports(1, &viewport);

    // Reset render targets to the screen.
    ID3D11RenderTargetView *const targets[1] = { deviceResources->GetBackBufferRenderTargetView() };
    context->OMSetRenderTargets(1, targets, deviceResources->GetDepthStencilView());

    // Clear the back buffer and depth stencil view.
    ID3D11RenderTargetView *targetView = deviceResources->GetBackBufferRenderTargetView();
    context->ClearRenderTargetView(targetView, DirectX::Colors::CornflowerBlue);
    context->ClearDepthStencilView(deviceResources->GetDepthStencilView(), D3D11_CLEAR_DEPTH | D3D11_CLEAR_STENCIL, 1.0f, 0);

    // Render the scene objects.
    m_sceneRenderer->Render();

    for (auto & renderer : m_renderers) {
        renderer->Render();
    }

    return true;
}

// Notifies renderers that device resources need to be released.
void nativeplayerMain::OnDeviceLost()
{
    m_sceneRenderer->ReleaseDeviceDependentResources();

    for (auto & renderer : m_renderers) {
        renderer->ReleaseDeviceDependentResources();
    }
}

// Notifies renderers that device resources may now be recreated.
void nativeplayerMain::OnDeviceRestored()
{
    m_sceneRenderer->CreateDeviceDependentResources();

    for (auto & renderer : m_renderers) {
        renderer->CreateDeviceDependentResources();
    }

    CreateWindowSizeDependentResources();
}
