#include "pch.h"
#include "../Shared/Internal.h"

using namespace std::literals;
using namespace Microsoft::VisualStudio::CppUnitTestFramework;
using namespace Twitch;

namespace WindowsLibTest {
	TEST_CLASS(DataSourceTest) {
public:
	TEST_METHOD(Connect_Success) {
		auto future = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		dataSource.Connect(configuration);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  \"connected\"\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(Connect_ImmediateTokenSuccess) {
		auto future = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		configuration.token.clear();
		configuration.onTokenExpired = [](DataSource::TokenRefreshFn fn) {
			fn(_T("token"));
			return true;
		};
		dataSource.Connect(configuration);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  \"connected\"\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(Connect_DelayedTokenSuccess) {
		auto future = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		configuration.token.clear();
		std::future<void> task;
		configuration.onTokenExpired = [&task](DataSource::TokenRefreshFn fn) {
			task = std::async(std::launch::async, [fn] {
				std::this_thread::sleep_for(11ms);
				fn(_T("token"));
			});
			return true;
		};
		dataSource.Connect(configuration);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  \"connected\"\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(Connect_TokenFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		configuration.token = _T("bad-token");
		std::promise<void> promise;
		configuration.onTokenExpired = [&promise](DataSource::TokenRefreshFn) {
			promise.set_value();
			return false;
		};
		dataSource.Connect(configuration);
		Assert::IsTrue(promise.get_future().wait_for(1110ms) != std::future_status::timeout);
	}

	TEST_METHOD(Connect_TwiceFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource, &configuration] {
			dataSource.Connect(configuration);
		});
	}

	TEST_METHOD(Connect_ParseFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		configuration.initialState = _T("$");
		Assert::ExpectException<TwitchException>([&dataSource, &configuration] {
			dataSource.Connect(configuration);
		});
	}

	TEST_METHOD(Connect_NonObjectFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		configuration.initialState = _T("1");
		Assert::ExpectException<TwitchException>([&dataSource, &configuration] {
			dataSource.Connect(configuration);
		});
	}

	TEST_METHOD(Connect_SizeFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		configuration.initialState = _T("[\"") + tstring(111'111, _T('a')) + _T("\"]");
		Assert::ExpectException<TwitchException>([&dataSource, &configuration] {
			dataSource.Connect(configuration);
		});
	}

	TEST_METHOD(Connect_InvalidMetadataFailure) {
		DataSource dataSource;
		tstring initialState = _T("{\"_metadata\":1}");
		DataSource::Configuration configuration = CreateConfiguration(initialState);
		Assert::ExpectException<TwitchException>([&dataSource, &configuration] {
			dataSource.Connect(configuration);
		});
	}

	TEST_METHOD(Connect_InvalidMetadataActiveFailure) {
		DataSource dataSource;
		tstring initialState = _T("{\"_metadata\":{\"active\":1}}");
		DataSource::Configuration configuration = CreateConfiguration(initialState);
		Assert::ExpectException<TwitchException>([&dataSource, &configuration] {
			dataSource.Connect(configuration);
		});
	}

	TEST_METHOD(Connect_InvalidMetadataIdFailure) {
		DataSource dataSource;
		tstring initialState = _T("{\"_metadata\":{\"id\":1}}");
		DataSource::Configuration configuration = CreateConfiguration(initialState);
		Assert::ExpectException<TwitchException>([&dataSource, &configuration] {
			dataSource.Connect(configuration);
		});
	}

	TEST_METHOD(Component_InvalidFailure) {
		auto future = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		configuration.token = _T("invalid-json");
		dataSource.Connect(configuration);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] invalid JSON from server:  expected value, got '$' (36)\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(Component_UnexpectedFailure) {
		auto future = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		configuration.token = _T("unexpected-json");
		dataSource.Connect(configuration);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  1\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(Component_AbortSuccess) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		configuration.token = _T("abort");
		std::promise<void> promise;
		configuration.onTokenExpired = [&promise](DataSource::TokenRefreshFn) {
			promise.set_value();
			return false;
		};
		dataSource.Connect(configuration);
		promise.get_future().get();
	}

	TEST_METHOD(Component_LongSuccess) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		configuration.token = _T("restart-long");
		std::promise<void> promise;
		configuration.onTokenExpired = [&promise](DataSource::TokenRefreshFn) {
			promise.set_value();
			return false;
		};
		dataSource.Connect(configuration);
		promise.get_future().get();
	}

	TEST_METHOD(Component_ShortSuccess) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		configuration.token = _T("restart-short");
		std::promise<void> promise;
		configuration.onTokenExpired = [&promise](DataSource::TokenRefreshFn) {
			promise.set_value();
			return false;
		};
		dataSource.Connect(configuration);
		promise.get_future().get();
	}

	TEST_METHOD(AppendToArrayField_Success) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":[1]}"));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto path = tstring(_T("a"));
		auto value = tstring(_T("[2]"));
		dataSource.AppendToArrayField(path, value);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"delta\":[[\"") +
			path + _T("\",\"a\",") + value + _T("]]}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(AppendToArrayField_EmptySuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":[1]}"));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		dataSource.AppendToArrayField(_T("a"), _T("[]"));
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSource::AppendToArrayField] warning:  array is empty; ignoring\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(AppendToArrayField_DoubleSuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":[1]}"));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto path = tstring(_T("a"));
		auto value1 = tstring(_T("[2]"));
		auto value2 = tstring(_T("[3]"));
		dataSource.AppendToArrayField(path, value1);
		dataSource.AppendToArrayField(path, value2);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"delta\":[[\"") +
			path + _T("\",\"a\",") + value1.substr(0, value1.size() - 1) + _T(',') + value2.substr(1) + _T("]]}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(AppendToArrayField_ThenRemoveSuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":[1]}"));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto path = tstring(_T("a"));
		dataSource.AppendToArrayField(path, _T("[2]"));
		dataSource.RemoveField(path);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"delta\":[[\"") +
			path + _T("\"]]}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(AppendToArrayField_NoConnectionFailure) {
		DataSource dataSource;
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.AppendToArrayField(_T("a"), _T("[1]"));
		});
	}

	TEST_METHOD(AppendToArrayField_EmptyPathFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":[1]}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.AppendToArrayField(_T(""), _T("[2]"));
		});
	}

	TEST_METHOD(AppendToArrayField_InvalidPathFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":[1]}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.AppendToArrayField(_T("a[0"), _T("[2]"));
		});
	}

	TEST_METHOD(AppendToArrayField_NoFieldFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":[1]}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.AppendToArrayField(_T("b"), _T("[2]"));
		});
	}

	TEST_METHOD(AppendToArrayField_NotArrayFieldFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":1}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.AppendToArrayField(_T("a"), _T("[2]"));
		});
	}

	TEST_METHOD(AppendToArrayField_NotArrayValueFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":[1]}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.AppendToArrayField(_T("a"), _T("2"));
		});
	}

	TEST_METHOD(AppendToArrayField_InvalidValueFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":[1]}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.AppendToArrayField(_T("a"), _T("$"));
		});
	}

	TEST_METHOD(RemoveField_SimpleSuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":1}"));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto path = tstring(_T("a"));
		dataSource.RemoveField(path);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"delta\":[[\"") +
			path + _T("\"]]}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(RemoveField_ComplexSuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"foo\":{\"bar\":[[{\"baz\":1},{\"baz\":2}]]}}"));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto path = tstring(_T("foo.bar[0][1].baz"));
		dataSource.RemoveField(path);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"delta\":[[\"") +
			path + _T("\"]]}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(RemoveField_DifferentSuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		auto path1 = tstring(_T("a"));
		auto path2 = tstring(_T("b"));
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"") + path1 + _T("\":1,\"") + path2 + _T("\":2}"));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		dataSource.RemoveField(path1);
		dataSource.RemoveField(path2);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"delta\":[[\"") +
			path1 + _T("\"],[\"") + path2 + _T("\"]]}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(RemoveField_ConsecutiveFailure) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		auto path = tstring(_T("a"));
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"") + path + _T("\":1}"));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		dataSource.RemoveField(path);
		Assert::ExpectException<TwitchException>([&dataSource, &path] {
			dataSource.RemoveField(path);
		});
		auto future = CreateErrorPromise();
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"delta\":[[\"") +
			path + _T("\"]]}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(RemoveField_ThenReplaceStateSuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		auto path = tstring(_T("a"));
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"") + path + _T("\":1}"));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		dataSource.RemoveField(path);
		auto state = AddMetadata(_T("{\"") + path + _T("\":1}"));
		dataSource.ReplaceState(state);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"refresh\":{\"data\":") +
			state + _T("}}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(RemoveField_ThenUpdateFieldSuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		auto path = tstring(_T("a"));
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"") + path + _T("\":1}"));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		dataSource.RemoveField(path);
		auto value = tstring(_T("2"));
		dataSource.UpdateField(path, value);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"delta\":[[\"") +
			path + _T("\",") + value + _T("]]}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(RemoveField_NoConnectionFailure) {
		DataSource dataSource;
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.RemoveField(_T("a"));
		});
	}

	TEST_METHOD(RemoveField_EmptyFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":1}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.RemoveField(_T(""));
		});
	}

	TEST_METHOD(RemoveField_ArrayReferenceFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":[1]}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.RemoveField(_T("a[0]"));
		});
	}

	TEST_METHOD(RemoveField_MetadataFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.RemoveField(_T("_metadata"));
		});
	}

	TEST_METHOD(RemoveField_InvalidPathFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":1}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.RemoveField(_T("a[0"));
		});
	}

	TEST_METHOD(RemoveField_NoFieldFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.RemoveField(_T("a"));
		});
	}

	TEST_METHOD(ReplaceState_Success) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":1}"));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto state = AddMetadata(_T("{\"b\":2}"));
		dataSource.ReplaceState(state);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"refresh\":{\"data\":") +
			state + _T("}}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(ReplaceState_DoubleSuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":1}"));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		dataSource.ReplaceState(_T("{\"b\":2}"));
		auto state = AddMetadata(_T("{\"c\":3}"));
		dataSource.ReplaceState(state);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"refresh\":{\"data\":") +
			state + _T("}}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(ReplaceState_ThenRemoveSuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":1}"));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto state = AddMetadata(_T("{\"b\":2}"));
		dataSource.ReplaceState(state.substr(0, state.size() - 1) + _T(",\"c\":3}"));
		dataSource.RemoveField(_T("c"));
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"refresh\":{\"data\":") +
			state + _T("}}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(ReplaceState_ThenUpdateSuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":1}"));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto state = AddMetadata(_T("{\"b\":2,\"c\":3}"));
		dataSource.ReplaceState(state.substr(0, state.size() - 2) + _T("4}"));
		dataSource.UpdateField(_T("c"), _T("3"));
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"refresh\":{\"data\":") +
			state + _T("}}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(ReplaceState_NoConnectionFailure) {
		DataSource dataSource;
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.ReplaceState(_T("{\"a\":1}"));
		});
	}

	TEST_METHOD(ReplaceState_ParseFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.ReplaceState(_T("$"));
		});
	}

	TEST_METHOD(ReplaceState_NotObjectFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.ReplaceState(_T("1"));
		});
	}

	TEST_METHOD(UpdateField_SimpleSuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":1}"));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto path = tstring(_T("a"));
		auto value = tstring(_T("2"));
		dataSource.UpdateField(path, value);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"delta\":[[\"") +
			path + _T("\",") + value + _T("]]}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(UpdateField_ComplexSuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"foo\":{\"bar\":[[{\"baz\":1},{\"baz\":2}]]}}"));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto path = tstring(_T("foo.bar[0][1].baz"));
		auto value = tstring(_T("3"));
		dataSource.UpdateField(path, value);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"delta\":[[\"") +
			path + _T("\",") + value + _T("]]}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(UpdateField_ArraySuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"foo\":{\"bar\":[[{\"baz\":1},{\"baz\":2}]]}}"));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto path = tstring(_T("foo.bar[0]"));
		auto value = tstring(_T("3"));
		dataSource.UpdateField(path, value);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"delta\":[[\"") +
			path + _T("\",") + value + _T("]]}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(UpdateField_DoubleSuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto path = tstring(_T("a"));
		dataSource.UpdateField(path, _T("1"));
		auto value = tstring(_T("2"));
		dataSource.UpdateField(path, value);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"delta\":[[\"") +
			path + _T("\",") + value + _T("]]}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(UpdateField_DifferentSuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto path1 = tstring(_T("a"));
		auto value1 = tstring(_T("1"));
		dataSource.UpdateField(path1, value1);
		auto path2 = tstring(_T("b"));
		auto value2 = tstring(_T("2"));
		dataSource.UpdateField(path2, value2);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"delta\":[[\"") +
			path1 + _T("\",") + value1 + _T("],[\"") + path2 + _T("\",") + value2 + _T("]]}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(UpdateField_ThenReplaceSuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		dataSource.UpdateField(_T("a"), _T("1"));
		auto state = AddMetadata(_T("{\"b\":2}"));
		dataSource.ReplaceState(state);
		AwaitErrorFuture(future);
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"refresh\":{\"data\":") +
			state + _T("}}\n"));
		Assert::AreEqual(expected, future.get());
	}

	TEST_METHOD(UpdateField_AsRefreshSuccess) {
		auto connectFuture = CreateErrorPromise();
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(AddMetadata(DataSource::EmptyState));
		dataSource.Connect(configuration);
		AwaitErrorFuture(connectFuture);
		auto future = CreateErrorPromise();
		auto path = tstring(_T("a"));
		auto value = _T('"') + tstring(5555, _T('a')) + _T('"');
		dataSource.UpdateField(path, value);
		AwaitErrorFuture(future);
		auto state = AddMetadata(_T("{\"") + path + _T("\":") + value + _T("}"));
		tstring expected(_T("[DataSourceImpl::ProcessMessage] unexpected JSON from server:  {\"refresh\":{\"data\":") +
			state + _T("}}\n"));
		auto got = future.get();
		Assert::AreEqual(expected, got);
	}

	TEST_METHOD(UpdateField_NoConnectionFailure) {
		DataSource dataSource;
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateField(_T("a"), _T("2"));
		});
	}

	TEST_METHOD(UpdateField_EmptyFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":1}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateField(_T(""), _T("2"));
		});
	}

	TEST_METHOD(UpdateField_ParseFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":1}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateField(_T("a"), _T("$"));
		});
	}

	TEST_METHOD(UpdateField_InvalidMetadataFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":1}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateField(_T("_metadata"), _T("1"));
		});
	}

	TEST_METHOD(UpdateField_InvalidMetadataActiveFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":1}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateField(_T("_metadata.active"), _T("1"));
		});
	}

	TEST_METHOD(UpdateField_InvalidMetadataIdFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":1}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateField(_T("_metadata.id"), _T("1"));
		});
	}

	TEST_METHOD(UpdateField_InvalidPathFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":1}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateField(_T("a[0"), _T("1"));
		});
	}

	TEST_METHOD(UpdateField_NoFieldFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration();
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateField(_T("a.b"), _T("1"));
		});
	}

	TEST_METHOD(UpdateField_OutOfBoundsFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"a\":[1]}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateField(_T("a[1]"), _T("1"));
		});
	}

	TEST_METHOD(UpdateField_ArrayFailure) {
		DataSource dataSource;
		DataSource::Configuration configuration = CreateConfiguration(_T("{\"foo\":{\"bar\":[[{\"baz\":1},{\"baz\":2}]]}}"));
		dataSource.Connect(configuration);
		Assert::ExpectException<TwitchException>([&dataSource] {
			dataSource.UpdateField(_T("foo.bar[1]"), _T("3"));
		});
	}

private:
	DataSource::Configuration CreateConfiguration(tstring const& initialState = DataSource::EmptyState) {
		DataSource::Configuration configuration;
		configuration.broadcasterIds = { _T("1") };
		configuration.environment = _T("test");
		configuration.gameId = _T("1");
		configuration.initialState = initialState;
		configuration.token = _T("1");
		configuration.onTokenExpired = [](DataSource::TokenRefreshFn) {
			Assert::Fail(_T("unexpected token refresh call-back"));
			return false;
		};
		return configuration;
	}

	static std::future<tstring> CreateErrorPromise() {
		static std::promise<tstring> promise;
		promise = std::promise<tstring>();
		struct Local {
			static void OutputDebugString(LPCTSTR s) {
				promise.set_value(s);
				test_OutputDebugString = nullptr;
			}
		};
		test_OutputDebugString = &Local::OutputDebugString;
		return promise.get_future();
	}

	static tstring AddMetadata(tstring const& state) {
		assert(!state.empty());
		assert(state.front() == _T('{'));
		assert(state.back() == _T('}'));
		auto metadata = tstring(_T("{\"_metadata\":{\"active\":true,\"id\":\"test\"}"));
		return state == DataSource::EmptyState ? metadata + _T('}') : metadata + _T(',') + state.substr(1);
	}

	void AwaitErrorFuture(std::future<tstring>& future) {
		auto result = future.wait_for(1110ms);
		test_OutputDebugString = nullptr;
		Assert::IsTrue(result != std::future_status::timeout);
	}
	};
}
