#include "pch.h"
#include "../Shared/Internal.h"
#include "Utilities.h"

using namespace Microsoft::VisualStudio::CppUnitTestFramework;
using namespace Twitch;
using namespace Utilities;

namespace Microsoft {
#if defined(_WIN32) && !defined(_XBOX_ONE)
	namespace VisualStudio {
		namespace CppUnitTestFramework {
			template<>
			static std::wstring ToString<int64_t>(int64_t const& t) {
				RETURN_WIDE_STRING(t);
			}
			template<>
			static std::wstring ToString<Stream::StreamType>(Stream::StreamType const& t) {
				switch(t) {
				case Stream::StreamType::Live:
					return L"Stream::StreamType::Live";
				case Stream::StreamType::None:
					return L"Stream::StreamType::None";
				case Stream::StreamType::Vodcast:
					return L"Stream::StreamType::Vodcast";
				default:
					return L"Unknown Stream::StreamType";
				}
			}
		}
	}
#endif
}

namespace WindowsLibTest {
	TEST_CLASS(StreamTest) {
public:
	TEST_METHOD(FetchDefaultStreams_Success) {
		StreamsRequest streamsRequest;
		auto streamResponse = streamsRequest.Fetch(clientId).get();
		Assert::AreNotEqual(tstring(), streamResponse.Cursor);
		Assert::IsTrue(streamResponse.Streams.size() >= 0ull);

		for(auto const& stream : streamResponse.Streams) {
			Assert::IsTrue(stream.CommunityIds.size() >= 0ull);
			Assert::AreNotEqual(tstring(), stream.Id);
			Assert::AreNotEqual(tstring(), stream.Language);
			Assert::AreNotEqual(time_t(), stream.StartedAt);
			Assert::AreNotEqual(tstring(), stream.ThumbnailUrl);
			Assert::IsTrue(stream.Type == Stream::StreamType::Live || stream.Type == Stream::StreamType::None || stream.Type == Stream::StreamType::Vodcast);
			Assert::AreNotEqual(tstring(), stream.UserId);
			Assert::IsTrue(stream.ViewerCount >= 1ull);
		}
	}

	TEST_METHOD(FetchOfflineUserStreams_Fail) {
		StreamsRequest streamsRequest;
		streamsRequest.UserLogin = login;
		auto streamResponse = streamsRequest.Fetch(clientId).get();
		Assert::AreEqual(tstring(), streamResponse.Cursor);
		Assert::AreEqual(0ull, streamResponse.Streams.size());
	}

	TEST_METHOD(FetchStreamByDisplayName_Success) {
		StreamsRequest streamsRequest;
		streamsRequest.Count = 1;
		auto streamResponse = streamsRequest.Fetch(clientId).get();
		Assert::AreNotEqual(tstring(), streamResponse.Cursor);
		Assert::IsTrue(streamResponse.Streams.size() <= 1ull);

		tstring topUserId = streamResponse.Streams[0].UserId;
		auto user = User::FetchOtherById(clientId, topUserId).get();
		streamsRequest.UserLogin = user.DisplayName;
		streamResponse = streamsRequest.Fetch(clientId).get();
		Assert::AreEqual(topUserId, streamResponse.Streams[0].UserId);
	}

	TEST_METHOD(FetchForwardThenBackward_Success) {
		StreamsRequest streamsRequest;
		streamsRequest.Count = 5;
		auto streamResponse = streamsRequest.Fetch(clientId).get();
		Assert::AreNotEqual(tstring(), streamResponse.Cursor);
		Assert::IsTrue(streamResponse.Streams.size() <= 5ull);

		streamsRequest.Cursor = streamResponse.Cursor;
		streamsRequest.Direction = StreamsRequest::PaginationDirection::Forward;
		streamResponse = streamsRequest.Fetch(clientId).get();
		Assert::AreNotEqual(tstring(), streamResponse.Cursor);
		Assert::IsTrue(streamResponse.Streams.size() <= 5ull);

		streamsRequest.Cursor = streamResponse.Cursor;
		streamsRequest.Direction = StreamsRequest::PaginationDirection::Backward;
		streamResponse = streamsRequest.Fetch(clientId).get();
		Assert::AreNotEqual(tstring(), streamResponse.Cursor);
		Assert::IsTrue(streamResponse.Streams.size() <= 5ull);
	}

	TEST_METHOD(FetchBackward_Fail) {
		StreamsRequest streamsRequest;
		streamsRequest.Count = 5;
		auto streamResponse = streamsRequest.Fetch(clientId).get();
		Assert::AreNotEqual(tstring(), streamResponse.Cursor);
		Assert::IsTrue(streamResponse.Streams.size() <= 5ull);

		streamsRequest.Cursor = streamResponse.Cursor;
		streamsRequest.Direction = StreamsRequest::PaginationDirection::Backward;
		Assert::ExpectException<TwitchException>([&streamsRequest] { streamsRequest.Fetch(clientId).get(); });
	}

	TEST_METHOD(FetchTopStream_Success) {
		StreamsRequest streamsRequest;
		streamsRequest.Count = 1;
		auto streamResponse = streamsRequest.Fetch(clientId).get();
		Assert::AreNotEqual(tstring(), streamResponse.Cursor);
		Assert::IsTrue(streamResponse.Streams.size() <= 1ull);


		streamsRequest.UserId = streamResponse.Streams[0].UserId;
		streamsRequest.Count = 0;
		auto streamResponse2 = streamsRequest.Fetch(clientId).get();

		Assert::AreEqual(streamResponse2.Cursor, streamResponse.Cursor);
		Assert::AreEqual(streamResponse2.Streams.size(), streamResponse.Streams.size());
		Assert::AreEqual(streamResponse2.Streams[0].CommunityIds.size(), streamResponse.Streams[0].CommunityIds.size());
		Assert::AreEqual(streamResponse2.Streams[0].GameId, streamResponse.Streams[0].GameId);
		Assert::AreEqual(streamResponse2.Streams[0].Id, streamResponse.Streams[0].Id);
		Assert::AreEqual(streamResponse2.Streams[0].Language, streamResponse.Streams[0].Language);
		Assert::AreEqual(streamResponse2.Streams[0].StartedAt, streamResponse.Streams[0].StartedAt);
		Assert::AreEqual(streamResponse2.Streams[0].ThumbnailUrl, streamResponse.Streams[0].ThumbnailUrl);
		Assert::AreEqual(streamResponse2.Streams[0].Title, streamResponse.Streams[0].Title);
		Assert::AreEqual(streamResponse2.Streams[0].Type, streamResponse.Streams[0].Type);
		Assert::AreEqual(streamResponse2.Streams[0].UserId, streamResponse.Streams[0].UserId);
	}
	};
}
