#include "pch.h"
#include "../Shared/Internal.h"
#include "Utilities.h"

using namespace Microsoft::VisualStudio::CppUnitTestFramework;
using namespace Twitch;
using namespace Utilities;

namespace {
	string_t const currentBio = _T("T-W-I-G real bio from API");
	string_t const updatedBio = _T("T-W-I-G real updated bio from API");

	bool IsServerFailure(int responseCode) {
		return responseCode >= 500 && responseCode < 600;
	}
}

namespace WindowsLibTest {
	TEST_CLASS(UserTest) {
public:
	TEST_METHOD(FetchCurrentUser_Success) {
		try {
			auto user = User::FetchCurrent(clientId, token).get();
			AssertTwiggieg(user, true);
			auto data = user.FetchProfileImage().get();
			Assert::AreEqual(62086ull, data.size());
			data = user.FetchOfflineImage().get();
			Assert::IsTrue(data.size() >= 0x8000ull);
		} catch(TwitchException const& ex) {
			if(ex.ErrorCode == 400) {
				Utilities::WriteWarning("ignoring user authentication error; test incomplete");
			} else {
				throw;
			}
		}
	}

	TEST_METHOD(FetchCurrentUser_Failure) {
		auto task = User::FetchCurrent(clientId, token_Invalid);
		Assert::ExpectException<TwitchException>([&task] { task.get(); });
	}

	TEST_METHOD(FetchOtherUserByLogin_Success) {
		auto user = User::FetchOtherByLogin(clientId_NoOAuth, login).get();
		AssertTwiggieg(user);
		auto userFromDisplayName = User::FetchOtherByLogin(clientId_NoOAuth, displayName).get();
		AssertTwiggieg(userFromDisplayName);
	}

	TEST_METHOD(FetchOtherUserById_Success) {
		auto user = User::FetchOtherById(clientId_NoOAuth, userId).get();
		AssertTwiggieg(user);
	}

	TEST_METHOD(FetchOtherUserByLogin_Failure) {
		auto task = User::FetchOtherByLogin(clientId_NoOAuth, invalidLogin);
		Assert::ExpectException<TwitchException>([&task] { task.get(); });
	}

	TEST_METHOD(FetchOtherUserById_Failure) {
		auto task = User::FetchOtherById(clientId_NoOAuth, invalidUserId);
		Assert::ExpectException<TwitchException>([&task] { task.get(); });
	}

	TEST_METHOD(FetchUserProfileImage_NoImage_Failure) {
		// This test hacks the user's profile image with a non-existent one.
		User user;
		auto* p = reinterpret_cast<tstring*>(&user);
		(p + 7)->assign(_T("https://static-cdn.jtvnw.net/jtv_user_pictures/f5955ee4e093f0c0-profile_image-457x219.png"));
		auto task = user.FetchProfileImage();
		Assert::ExpectException<TwitchException>([&task] { task.get(); });
	}

	TEST_METHOD(FetchUserProfileImage_InvalidUrl_Failure) {
		// This test hacks the user's profile image with an invalid one.
		User user;
		auto* p = reinterpret_cast<tstring*>(&user);
		(p + 7)->assign(_T("invalid:url"));
		auto task = user.FetchProfileImage();
		Assert::ExpectException<TwitchException>([&task] { task.get(); });
	}

	TEST_METHOD(UpdateUserDescription_Success) {
		try {
			auto user = User::FetchCurrent(clientId, token).get();
			user.UpdateDescription(clientId, token, updatedBio).get();
			user = User::FetchCurrent(clientId, token).get();
			Assert::AreEqual(tstring(updatedBio), user.Description);
			user.UpdateDescription(clientId, token, currentBio).get();
		} catch(TwitchException const& ex) {
			if(ex.ErrorCode == 400) {
				Utilities::WriteWarning("ignoring user authentication error; test incomplete");
			} else if(IsServerFailure(ex.ErrorCode)) {
				Utilities::WriteWarning("ignoring server failure; test incomplete");
			} else {
				throw;
			}
		}
	}

	TEST_METHOD(GetFollowers_Success) {
		// This user has over two million followers.
		tstring popularUserId = _T("23161357");

		// Get the first page, default size, of followers.  Include a token.
		FollowersRequest request;
		auto response = request.Fetch(clientId, token, popularUserId).get();
		Assert::IsTrue(response.TotalCount >= 2'000'000ull);
		auto firstUserId = response.Followers.front().UserId;

		// Get the next page, larger size, of followers.  Don't include a token.
		request.Count = 30;
		request.Cursor = response.Cursor;
		request.Direction = FollowersRequest::PaginationDirection::Forward;
		response = request.Fetch(clientId, tstring(), popularUserId).get();
		Assert::AreEqual(static_cast<decltype(response.Followers.size())>(request.Count), response.Followers.size());
		Assert::IsTrue(response.TotalCount >= 2'000'000ull);
		Assert::AreNotEqual(firstUserId, response.Followers.front().UserId);
	}

	TEST_METHOD(CreateClip_Failure) {
		auto task = Clip::Create(clientId, token, userId);
		int resultCode = 0;
		try {
			task.get();
		} catch(TwitchException const& ex) {
			resultCode = ex.ErrorCode;
		}
		Assert::AreEqual(404, resultCode);
	}

	TEST_METHOD(FollowUser_Success) {
		try {
			auto user = User::FetchCurrent(clientId, token).get();
			auto result = user.FollowChannel(clientId, token, _T("129454141")).get();
			Assert::IsTrue(result);
		} catch(TwitchException const& ex) {
			if(ex.ErrorCode == 400) {
				Utilities::WriteWarning("ignoring user authentication error; test incomplete");
			} else if(IsServerFailure(ex.ErrorCode)) {
				Utilities::WriteWarning("ignoring server failure; test incomplete");
			} else {
				throw;
			}
		}
	}

	TEST_METHOD(FollowUser_InvalidChannel) {
		try {
			auto user = User::FetchCurrent(clientId, token).get();
			auto result = user.FollowChannel(clientId, token, _T("notachannel")).get();
			Assert::IsFalse(result);
		} catch(TwitchException const& ex) {
			if(ex.ErrorCode == 400) {
				Utilities::WriteWarning("ignoring user authentication error; test incomplete");
			} else if(IsServerFailure(ex.ErrorCode)) {
				Utilities::WriteWarning("ignoring server failure; test incomplete");
			} else {
				throw;
			}
		}
	}

	TEST_METHOD(FollowUser_NoChannel) {
		try {
			auto user = User::FetchCurrent(clientId, token).get();
			auto result = user.FollowChannel(clientId, token, _T("000000000")).get();
			Assert::IsFalse(result);
		} catch(TwitchException const& ex) {
			if(ex.ErrorCode == 400) {
				Utilities::WriteWarning("ignoring user authentication error; test incomplete");
			} else if(IsServerFailure(ex.ErrorCode)) {
				Utilities::WriteWarning("ignoring server failure; test incomplete");
			} else {
				throw;
			}
		}
	}

	TEST_METHOD(FollowUser_SelfChannel) {
		try {
			auto user = User::FetchCurrent(clientId, token).get();
			auto result = user.FollowChannel(clientId, token, userId).get();
			Assert::IsFalse(result);
		} catch(TwitchException const& ex) {
			if(ex.ErrorCode == 400) {
				Utilities::WriteWarning("ignoring user authentication error; test incomplete");
			} else if(IsServerFailure(ex.ErrorCode)) {
				Utilities::WriteWarning("ignoring server failure; test incomplete");
			} else {
				throw;
			}
		}
	}

	private:
	static void AssertTwiggieg(User const& user, bool isAuthenticated = false) {
		Assert::AreEqual(tstring(), user.BroadcasterType);
		if(user.Description != currentBio) {
			Utilities::WriteWarning("ignoring user description mis-match; test incomplete");
		}
		Assert::AreEqual(tstring(_T("TwIGgIEg")), user.DisplayName);
		if(isAuthenticated) {
			Assert::AreEqual(tstring(_T("loohill@twitch.tv")), user.Email);
		} else {
			Assert::AreEqual(tstring(), user.Email);
			Assert::AreNotEqual(tstring(_T("loohill@twitch.tv")), user.Email);
		}
		Assert::AreEqual(tstring(_T("twiggieg")), user.Login);
		Assert::AreEqual(tstring(), user.OfflineImageUrl);
		Assert::AreEqual(tstring(_T("https://static-cdn.jtvnw.net/jtv_user_pictures/f5955ee4e093f0c0-profile_image-300x300.png")), user.ProfileImageUrl);
		Assert::AreEqual(tstring(), user.Type);
		Assert::AreEqual(tstring(userId), user.Id);
		Assert::IsTrue(user.ViewCount >= 1ull);
	}
	};
}
