struct WebSocket::WebSocketImpl {
	// Do not invoke Open or delete the WebSocketImpl instance in the OnClosed
	// handler or a deadlock will occur.  Do not invoke Close, Open, or delete the
	// WebSocketImpl instance in the OnReceived handler or a deadlock will occur.

	ClosedFn OnClosed = DefaultClosedFn;
	ReceivedFn OnReceived = DefaultReceivedFn;
	bool GetIsOpen() const { return handle.IsOpen; }
	__declspec(property(get = GetIsOpen)) bool const IsOpen;

	WebSocketImpl() = default;
	WebSocketImpl(WebSocketImpl const&) = delete;
	WebSocketImpl(WebSocketImpl&&) = default;
	WebSocketImpl& operator=(WebSocketImpl const&) = delete;
	WebSocketImpl& operator=(WebSocketImpl&&) = default;
	~WebSocketImpl() {
		Close();
	}

	void Close() {
		if(handle.Close()) {
			SafelyInvoke(OnClosed);
			receiveTask.get();
		}
	}

	int Open(string_t url, std::chrono::milliseconds timeout) {
		if(IsOpen) {
			DebugWriteLine(_T("[WebSocketImpl::Open] warning:  Open called without preceding Close"));
			assert(!IsOpen);
			Close();
		}
		int errorCode;
		uintptr_t handle_;
		std::tie(errorCode, handle_) = Platform::StartHttpRequest(url, HTTP_WEB_SOCKET_REQUEST,
			timeout, std::vector<HttpParam>(), HttpRequest::EmptyBody);
		if(!errorCode) {
			handle = SynchronizedHandle(handle_);
			auto webSocket = Internets::GetHttpRequest(handle_);
			receiveTask = std::async(std::launch::async, [this, webSocket] { ReceiveMessages(webSocket); });
		}
		return errorCode;
	}

	int Send(void const* data, size_t size) {
		auto webSocket = handle.Get();
		auto errorCode = WinHttpWebSocketSend(webSocket,
			WINHTTP_WEB_SOCKET_BINARY_MESSAGE_BUFFER_TYPE,
			const_cast<PVOID>(data), static_cast<DWORD>(size));
		if(errorCode != ERROR_SUCCESS) {
			return FromPlatformError(errorCode);
		}
		return 0;
	}

private:
	struct SynchronizedHandle {
		explicit SynchronizedHandle(uintptr_t handle) : handle(handle) {}
		SynchronizedHandle() = default;
		SynchronizedHandle(SynchronizedHandle const&) = delete;
		SynchronizedHandle(SynchronizedHandle&& that) noexcept { std::swap(handle, that.handle); }
		SynchronizedHandle& operator=(SynchronizedHandle const&) = delete;
		SynchronizedHandle& operator=(SynchronizedHandle&& that) noexcept {
			std::swap(handle, that.handle);
			return *this;
		}
		~SynchronizedHandle() {
			Close();
		}
		bool GetIsOpen() const { return static_cast<bool>(handle); }
		__declspec(property(get = GetIsOpen)) bool const IsOpen;

		bool Close() {
			std::lock_guard<decltype(mutex)> lock(mutex);
			if(handle) {
				Platform::FinishHttpRequest(handle);
				return true;
			}
			return false;
		}
		HINTERNET Get() {
			std::lock_guard<decltype(mutex)> lock(mutex);
			return Internets::GetHttpRequest(handle);
		}

	private:
		std::mutex mutex;
		uintptr_t handle = 0;
	};

	std::future<void> receiveTask;
	SynchronizedHandle handle;

	void ReceiveMessages(HINTERNET webSocket) {
		while(handle.IsOpen) {
			std::vector<uint8_t> data;
			WINHTTP_WEB_SOCKET_BUFFER_TYPE bufferType;
			do {
				constexpr DWORD bufferDelta = 1024;
				data.resize(data.size() + bufferDelta);
				auto* p = &data.back() - bufferDelta + 1;
				DWORD n;
				auto errorCode = WinHttpWebSocketReceive(webSocket, p, bufferDelta, &n, &bufferType);
				if(errorCode != ERROR_SUCCESS) {
					if(handle.Close()) {
						DebugWriteLine(_T("[WebSocketImpl::ReceiveMessages] WinHttpWebSocketReceive error %d"), errorCode);
						SafelyInvoke(OnClosed);
					}
					return;
				}
				data.resize(data.size() - bufferDelta + n);
			} while(bufferType == WINHTTP_WEB_SOCKET_BINARY_FRAGMENT_BUFFER_TYPE || bufferType == WINHTTP_WEB_SOCKET_UTF8_FRAGMENT_BUFFER_TYPE);
			SafelyInvoke([this, &data]() mutable { OnReceived(data); });
		}
	}

	template<typename FN>
	void SafelyInvoke(FN fn) {
		try {
			fn();
		} catch(std::exception const& ex) {
			DebugWriteLine(_T("[WebSocketImpl::SafelyInvoke] function threw an exception \"%hs\""), ex.what());
			assert(false);
		}
	}
};

#include "WebSocket.inl"
