#include "pch.h"
#include <wrl.h>
#include "mediaplayer/player.hpp"
#include "mediaplayer/platforms/windows/PlayerCore.hpp"
#include "../platforms/windows/VideoSample.hpp"
#include "mediaplayer/platforms/windows/WindowsPlatform.hpp"

using namespace Twitch;

#include "ListenerImpl.inl"

namespace {
	twitch::MediaTime FromDouble(double time) {
		return time == std::numeric_limits<double>::infinity() ? twitch::MediaTime::max() : twitch::MediaTime(time);
	}

	double ToDouble(twitch::MediaTime const& time) {
		return time == twitch::MediaTime::max() ? std::numeric_limits<double>::infinity() : time.seconds();
	}
}

struct Player::PlayerImpl {
	PlayerImpl(Observer& observer, ID3D11Device1* device, RenderFn onFrame) : listener(observer), device(device), onFrame(onFrame) {
		auto onRenderedFrame = [onFrame](twitch::MediaSampleBuffer const& videoSample, int width, int height) {
			if(videoSample.type == twitch::windows::MediaSample::Type::Opaque) {
				onFrame(static_cast<twitch::windows::VideoSample const*>(&videoSample)->getTexture().Get(), width, height);
			}
		};

		player = twitch::windows::PlayerCore::create(listener, onRenderedFrame, device);
		player->setDeviceId(FromTstring(GetDeviceId()));
	}

	ListenerImpl listener;
	ID3D11Device1* device;
	RenderFn onFrame;
	std::shared_ptr<twitch::Player> player;
};

Player::Player(Observer& observer, ID3D11Device1* device, RenderFn onFrame) {
	pimpl = std::make_unique<PlayerImpl>(observer, device, onFrame);
}

Player::Player(Player&&) noexcept = default;
Player& Player::operator=(Player&&) noexcept = default;

Player::~Player() {}

void Player::Load(tstring const& url) {
	pimpl->player->load(FromTstring(url));
}

void Player::Play() {
	pimpl->player->play();
}

void Player::Pause() {
	pimpl->player->pause();
}

bool Player::GetIsSeekable() const {
	return pimpl->player->isSeekable();
}

double Player::GetDuration() const {
	return ToDouble(pimpl->player->getDuration());
}

double Player::GetPosition() const {
	return ToDouble(pimpl->player->getPosition());
}

void Player::SetPosition(double position) {
	pimpl->player->seekTo(FromDouble(position));
}

double Player::GetBufferedPosition() const {
	return ToDouble(pimpl->player->getBufferedPosition());
}

Player::PlayerState Player::GetState() const {
	return static_cast<PlayerState>(pimpl->player->getState());
}

float Player::GetPlaybackRate() const {
	return pimpl->player->getPlaybackRate();
}

void Player::SetPlaybackRate(float playbackRate) {
	pimpl->player->setPlaybackRate(playbackRate);
}

bool Player::GetIsLooping() const {
	return pimpl->player->isLooping();
}

void Player::SetIsLooping(bool isLooping) {
	pimpl->player->setLooping(isLooping);
}

bool Player::GetIsMuted() const {
	return pimpl->player->isMuted();
}

void Player::SetIsMuted(bool isMuted) {
	pimpl->player->setMuted(isMuted);
}

float Player::GetVolume() const {
	return pimpl->player->getVolume();
}

void Player::SetVolume(float volume) {
	pimpl->player->setVolume(volume);
}
