#include "pch.h"
#include "RawSocket.h"

#pragma comment(lib, "WS2_32.lib")

using namespace Twitch;

namespace {
	bool isInitialized;

	inline void ThrowIfError(int result) {
		if(result) {
			throw TwitchException(FromPlatformError(result));
		}
	}

	void CheckInitialization() {
		if(!isInitialized) {
			WSADATA wsaData;
			ThrowIfError(WSAStartup(MAKEWORD(2, 2), &wsaData));
			isInitialized = true;
		}
	}
}

RawSocket::RawSocket() {
	CheckInitialization();
	value = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
	if(value == INVALID_SOCKET) {
		throw TwitchException(FromPlatformError(WSAGetLastError()));
	}
}

RawSocket::~RawSocket() {
	Close();
}

void RawSocket::InternalClose() noexcept {
	closesocket(value);
}

int RawSocket::InternalConnect(string_t hostName, unsigned short port, std::chrono::milliseconds timeout /*= DefaultTimeout*/) noexcept {
	auto timeout_ = static_cast<DWORD>(timeout.count());
	if(setsockopt(value, SOL_SOCKET, SO_RCVTIMEO, reinterpret_cast<char const*>(&timeout_), sizeof(timeout_))) {
		return FromPlatformError(WSAGetLastError());
	}
	if(setsockopt(value, SOL_SOCKET, SO_SNDTIMEO, reinterpret_cast<char const*>(&timeout_), sizeof(timeout_))) {
		return FromPlatformError(WSAGetLastError());
	}
	sockaddr_in address{};
	if(InetPton(AF_INET, hostName, &address.sin_addr) <= 0) {
		ADDRINFOT hints{};
		hints.ai_family = AF_INET;
		hints.ai_protocol = IPPROTO_TCP;
		hints.ai_socktype = SOCK_STREAM;
		ADDRINFOT* addresses;
		auto errorCode = GetAddrInfo(hostName, nullptr, &hints, &addresses);
		if(errorCode) {
			return FromPlatformError(errorCode);
		}
		ADDRINFOT const* p;
		for(p = addresses; p != nullptr; p = p->ai_next) {
			if(p->ai_family == AF_INET && p->ai_protocol == IPPROTO_TCP && p->ai_socktype == SOCK_STREAM) {
				address.sin_addr = reinterpret_cast<sockaddr_in const*>(p->ai_addr)->sin_addr;
				break;
			}
		}
		FreeAddrInfo(addresses);
		if(p == nullptr) {
			return FromPlatformError(ERROR_INCORRECT_ADDRESS);
		}
	}
	address.sin_family = AF_INET;
	address.sin_port = htons(port);
	if(connect(value, reinterpret_cast<sockaddr const*>(&address), sizeof(address))) {
		return FromPlatformError(WSAGetLastError());
	}
	return 0;
}

int RawSocket::InternalReceive(void* buffer, size_t size) noexcept {
	int const result = recv(value, static_cast<char*>(buffer), static_cast<int>(size), 0);
	return result < 0 ? FromPlatformError(WSAGetLastError()) : result;
}

int RawSocket::InternalSend(void const* buffer, size_t size) noexcept {
	int const result = send(value, static_cast<char const*>(buffer), static_cast<int>(size), 0);
	return result < 0 ? FromPlatformError(WSAGetLastError()) : result;
}
