import createMDaaS, { Configuration, State, TokenRefreshFn } from '../lib/index';

test('connect succeeds', async () => {
  const promise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  await mdaas.connect(configuration);
  expect(await promise).toBe('"connected"');
  mdaas.disconnect();
});

test('connect succeeds with immediate token acquisition', async () => {
  const promise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  configuration.token = '';
  configuration.onTokenExpired = (fn: TokenRefreshFn) => {
    fn('token');
    return true;
  };
  await mdaas.connect(configuration);
  expect(await promise).toBe('"connected"');
  mdaas.disconnect();
});

test('connect succeeds with delayed token acquisition', async () => {
  const promise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  configuration.token = '';
  configuration.onTokenExpired = (fn: TokenRefreshFn) => {
    setTimeout(() => fn('token'), 11);
    return true;
  };
  await mdaas.connect(configuration);
  expect(await promise).toBe('"connected"');
  mdaas.disconnect();
});

test('connect fails due to token', async () => {
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  configuration.token = 'bad-token';
  const promise = createTokenExpiredPromise(configuration);
  await mdaas.connect(configuration);
  await promise;
  mdaas.disconnect();
});

test('connect fails due to current connection', async () => {
  expect.hasAssertions();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  await mdaas.connect(configuration);
  try {
    await mdaas.connect(configuration);
    console.error('unexpected success');
    mdaas.disconnect();
  } catch(ex) {
    expect(ex.message).toBe('already connected');
  }
  mdaas.disconnect();
});

test('connect fails due to non-object', async () => {
  expect.hasAssertions();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  configuration.initialState = '$' as any as State;
  try {
    await mdaas.connect(configuration);
    console.error('unexpected success');
    mdaas.disconnect();
  } catch(ex) {
    expect(ex.message).toBe('initial state is not an object');
  }
});

test('connect fails due to size', async () => {
  expect.hasAssertions();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  configuration.initialState.large = Array(99).fill(Array(999).fill('a').join('b'));
  try {
    await mdaas.connect(configuration);
    console.error('unexpected success');
    mdaas.disconnect();
  } catch(ex) {
    expect(ex.message).toBe('initial state is too large');
  }
});

test('connect fails due to invalid _metadata', async () => {
  expect.hasAssertions();
  const mdaas = createMDaaS();
  const initialState = { _metadata: 1 };
  const configuration = createConfiguration(initialState);
  try {
    await mdaas.connect(configuration);
    console.error('unexpected success');
    mdaas.disconnect();
  } catch(ex) {
    expect(ex.message).toBe('_metadata value is not an object');
  }
});

test('connect fails due to invalid _metadata.active', async () => {
  expect.hasAssertions();
  const mdaas = createMDaaS();
  const initialState = { _metadata: { active: 1 } };
  const configuration = createConfiguration(initialState);
  try {
    await mdaas.connect(configuration);
    console.error('unexpected success');
    mdaas.disconnect();
  } catch(ex) {
    expect(ex.message).toBe('_metadata.active value is not a Boolean');
  }
});

test('connect fails due to invalid _metadata.id', async () => {
  expect.hasAssertions();
  const mdaas = createMDaaS();
  const initialState = { _metadata: { id: 1 } };
  const configuration = createConfiguration(initialState);
  try {
    await mdaas.connect(configuration);
    console.error('unexpected success');
    mdaas.disconnect();
  } catch(ex) {
    expect(ex.message).toBe('_metadata.id value is not a string');
  }
});

test('component handles invalid JSON from server', async () => {
  const promise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  configuration.token = 'invalid-json';
  await mdaas.connect(configuration);
  expect(await promise).toBe('Unexpected token $ in JSON at position 0');
  mdaas.disconnect();
});

test('component handles unexpected JSON from server', async () => {
  const promise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  configuration.token = 'unexpected-json';
  await mdaas.connect(configuration);
  expect(await promise).toBe('1');
  mdaas.disconnect();
});

test('component aborts successfully', async () => {
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  configuration.token = 'abort';
  const promise = createTokenExpiredPromise(configuration);
  await mdaas.connect(configuration);
  await promise;
  mdaas.disconnect();
});

test('component reconnects after long wait', async () => {
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  configuration.token = 'restart-long';
  const promise = createTokenExpiredPromise(configuration, 4444);
  await mdaas.connect(configuration);
  await promise;
  mdaas.disconnect();
});

test('component reconnects after short wait', async () => {
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  configuration.token = 'restart-short';
  const promise = createTokenExpiredPromise(configuration);
  await mdaas.connect(configuration);
  await promise;
  mdaas.disconnect();
});

test('append succeeds', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: [1] });
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  const path = 'a';
  const value = [2];
  mdaas.appendToArrayField(path, value);
  expect(await promise).toBe(JSON.stringify({ delta: [[path, 'a', value]] }));
  mdaas.disconnect();
});

test('append empty succeeds', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: [1] });
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  const path = 'a';
  const value: any[] = [];
  mdaas.appendToArrayField(path, value);
  expect(await promise).toBe('[MDaaS.appendToArrayField] array is empty; ignoring');
  mdaas.disconnect();
});

test('double-append succeeds', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: [1] });
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  const path = 'a';
  const value1 = [2];
  mdaas.appendToArrayField(path, value1);
  const value2 = [3];
  mdaas.appendToArrayField(path, value2);
  expect(await promise).toBe(JSON.stringify({ delta: [[path, 'a', value1.concat(value2)]] }));
  mdaas.disconnect();
});

test('append then remove succeeds', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: [1] });
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  const path = 'a';
  mdaas.appendToArrayField(path, [2]);
  mdaas.removeField(path);
  expect(await promise).toBe(JSON.stringify({ delta: [[path]] }));
  mdaas.disconnect();
});

test('append fails due to no connection', async () => {
  expect.hasAssertions();
  const mdaas = createMDaaS();
  try {
    await mdaas.appendToArrayField('a', [1]);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe('connection not established');
  }
});

test('append fails due to empty path', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: [1] });
  await mdaas.connect(configuration);
  await connectionPromise;
  try {
    await mdaas.appendToArrayField('', [2]);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe('path is empty');
  }
  mdaas.disconnect();
});

test('append fails due to invalid path type', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: [1] });
  await mdaas.connect(configuration);
  await connectionPromise;
  try {
    await mdaas.appendToArrayField(1 as any as string, [2]);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe('path is not a string');
  }
  mdaas.disconnect();
});

test('append fails due to invalid path', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: [1] });
  await mdaas.connect(configuration);
  await connectionPromise;
  const path = 'a[0';
  try {
    mdaas.appendToArrayField(path, [2]);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe(`"${path}" is not a valid field specifier`);
  }
  mdaas.disconnect();
});

test('append fails due to no field', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: [1] });
  await mdaas.connect(configuration);
  await connectionPromise;
  const path = 'b';
  try {
    mdaas.appendToArrayField(path, [2]);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe(`"${path}" does not specify a known field`);
  }
  mdaas.disconnect();
});

test('append fails due to not an array field', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: 1 });
  await mdaas.connect(configuration);
  await connectionPromise;
  const path = 'a';
  try {
    mdaas.appendToArrayField(path, [2]);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe(`"${path}" does not specify an array field`);
  }
  mdaas.disconnect();
});

test('append fails due to not an array value', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: [1] });
  await mdaas.connect(configuration);
  await connectionPromise;
  const path = 'a';
  try {
    mdaas.appendToArrayField(path, 2 as any as any[]);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe('values is not an array');
  }
  mdaas.disconnect();
});

test('remove simple succeeds', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: 1 });
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  const path = 'a';
  mdaas.removeField(path);
  expect(await promise).toBe(JSON.stringify({ delta: [[path]] }));
  mdaas.disconnect();
});

test('remove complex succeeds', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ foo: { bar: [[{ baz: 1 }, { baz: 2 }]] } });
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  const path = 'foo.bar[0][1].baz';
  mdaas.removeField(path);
  expect(await promise).toBe(JSON.stringify({ delta: [[path]] }));
  mdaas.disconnect();
});

test('different removes succeed', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const path1 = 'a';
  const path2 = 'b';
  const configuration = createConfiguration({ [path1]: 1, [path2]: 2 });
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  mdaas.removeField(path1);
  mdaas.removeField(path2);
  expect(await promise).toBe(JSON.stringify({ delta: [[path1], [path2]] }));
  mdaas.disconnect();
});

test('first remove succeeds and second remove fails', async () => {
  expect.assertions(2);
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: 1 });
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  const path = 'a';
  mdaas.removeField(path);
  try {
    mdaas.removeField(path);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe(`"${path}" does not specify a known field`);
  }
  expect(await promise).toBe(JSON.stringify({ delta: [[path]] }));
  mdaas.disconnect();
});

test('remove then replace succeeds', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: 1 });
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  mdaas.removeField('a');
  const state = withMetadata({ b: 2 });
  mdaas.replaceState(state);
  expect(await promise).toBe(JSON.stringify({ refresh: { data: state } }));
  mdaas.disconnect();
});

test('remove then update succeeds', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: 1 });
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  const path = 'a';
  mdaas.removeField(path);
  const value = 2;
  mdaas.updateField(path, value);
  expect(await promise).toBe(JSON.stringify({ delta: [[path, value]] }));
  mdaas.disconnect();
});

test('remove fails due to no connection', async () => {
  expect.hasAssertions();
  const mdaas = createMDaaS();
  try {
    await mdaas.removeField('');
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe('connection not established');
  }
});

test('remove fails due to empty path', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: 1 });
  await mdaas.connect(configuration);
  await connectionPromise;
  try {
    mdaas.removeField('');
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe('path is empty');
  }
  mdaas.disconnect();
});

test('remove fails due to invalid type', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: 1 });
  await mdaas.connect(configuration);
  await connectionPromise;
  try {
    mdaas.removeField(1 as any as string);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe('path is not a string');
  }
  mdaas.disconnect();
});

test('remove fails due to array reference', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: [1] });
  await mdaas.connect(configuration);
  await connectionPromise;
  const path = 'a[0]';
  try {
    mdaas.removeField(path);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe(`"${path}" does not specify a field`);
  }
  mdaas.disconnect();
});

test('remove fails due to metadata reference', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  await mdaas.connect(configuration);
  await connectionPromise;
  const path = '_metadata';
  try {
    mdaas.removeField(path);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe(`cannot remove "${path}"`);
  }
  mdaas.disconnect();
});

test('remove fails due to invalid path', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  await mdaas.connect(configuration);
  await connectionPromise;
  const path = 'a[0';
  try {
    mdaas.removeField(path);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe(`"${path}" is not a valid field specifier`);
  }
  mdaas.disconnect();
});

test('remove fails due to no field', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  await mdaas.connect(configuration);
  await connectionPromise;
  const path = 'a';
  try {
    mdaas.removeField(path);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe(`"${path}" does not specify a known field`);
  }
  mdaas.disconnect();
});

test('replace succeeds', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: 1 });
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  const newState = withMetadata({ b: 2 });
  mdaas.replaceState(newState);
  expect(await promise).toBe(JSON.stringify({ refresh: { data: newState } }));
  mdaas.disconnect();
});

test('double-replace succeeds', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: 1 });
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  mdaas.replaceState({ b: 2 });
  const newState = withMetadata({ c: 3 });
  mdaas.replaceState(newState);
  expect(await promise).toBe(JSON.stringify({ refresh: { data: newState } }));
  mdaas.disconnect();
});

test('replace then remove succeeds', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: 1 });
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  const newState = withMetadata({ b: 2 });
  const path = 'c';
  mdaas.replaceState({ ...newState, [path]: 3 });
  mdaas.removeField(path);
  expect(await promise).toBe(JSON.stringify({ refresh: { data: newState } }));
  mdaas.disconnect();
});

test('replace then update succeeds', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: 1 });
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  const path = 'c';
  const value = 3;
  const newState = withMetadata({ b: 2 });
  mdaas.replaceState({ ...newState, c: ~value });
  mdaas.updateField(path, value);
  expect(await promise).toBe(JSON.stringify({ refresh: { data: { ...newState, [path]: value } } }));
  mdaas.disconnect();
});

test('replace fails due to no connection', async () => {
  expect.hasAssertions();
  const mdaas = createMDaaS();
  try {
    await mdaas.replaceState({});
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe('connection not established');
  }
});

test('replace fails due to not an object', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  await mdaas.connect(configuration);
  await connectionPromise;
  try {
    await mdaas.replaceState(1 as any as State);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe('state is not an object');
  }
  mdaas.disconnect();
});

test('update simple succeeds', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  const path = 'a';
  const value = 1;
  mdaas.updateField(path, value);
  expect(await promise).toBe(JSON.stringify({ delta: [[path, value]] }));
  mdaas.disconnect();
});

test('update complex succeeds', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ foo: { bar: [[{ baz: 1 }, { baz: 2 }]] } });
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  const path = 'foo.bar[0][1].baz';
  const value = 3;
  mdaas.updateField(path, value);
  expect(await promise).toBe(JSON.stringify({ delta: [[path, value]] }));
  mdaas.disconnect();
});

test('update array succeeds', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ foo: { bar: [[{ baz: 1 }, { baz: 2 }]] } });
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  const path = 'foo.bar[0]';
  const value = 3;
  mdaas.updateField(path, value);
  expect(await promise).toBe(JSON.stringify({ delta: [[path, value]] }));
  mdaas.disconnect();
});

test('double-update succeeds', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  const path = 'a';
  mdaas.updateField(path, 1);
  const value = 2;
  mdaas.updateField(path, value);
  expect(await promise).toBe(JSON.stringify({ delta: [[path, value]] }));
  mdaas.disconnect();
});

test('different updates succeed', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  const path1 = 'a';
  const value1 = 1;
  mdaas.updateField(path1, value1);
  const path2 = 'b';
  const value2 = 2;
  mdaas.updateField(path2, value2);
  expect(await promise).toBe(JSON.stringify({ delta: [[path1, value1], [path2, value2]] }));
  mdaas.disconnect();
});

test('update then replace succeeds', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  mdaas.updateField('a', 1);
  const newState = withMetadata({ b: 2 });
  mdaas.replaceState(newState);
  expect(await promise).toBe(JSON.stringify({ refresh: { data: newState } }));
  mdaas.disconnect();
});

test('update succeeds as refresh', async () => {
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration(withMetadata({}));
  await mdaas.connect(configuration);
  await connectionPromise;
  const promise = createConsolePromise();
  const path = 'a';
  const value = Array(999).fill('abc').join('def');
  const newState = withMetadata({ [path]: value });
  mdaas.updateField(path, value);
  expect(await promise).toBe(JSON.stringify({ refresh: { data: newState } }));
  mdaas.disconnect();
});

test('update fails due to no connection', async () => {
  expect.hasAssertions();
  const mdaas = createMDaaS();
  try {
    await mdaas.updateField('', {});
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe('connection not established');
  }
});

test('update fails due to empty path', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: 1 });
  await mdaas.connect(configuration);
  await connectionPromise;
  try {
    mdaas.updateField('', 2);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe('path is empty');
  }
  mdaas.disconnect();
});

test('update fails due to invalid type', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: 1 });
  await mdaas.connect(configuration);
  await connectionPromise;
  try {
    mdaas.updateField(1 as any as string, 2);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe('path is not a string');
  }
  mdaas.disconnect();
});

test('update fails due to invalid _metadata', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: 1 });
  await mdaas.connect(configuration);
  await connectionPromise;
  try {
    mdaas.updateField('_metadata', 1);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe('_metadata value is not an object');
  }
  mdaas.disconnect();
});

test('update fails due to invalid _metadata.active', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: 1 });
  await mdaas.connect(configuration);
  await connectionPromise;
  try {
    mdaas.updateField('_metadata.active', 1);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe('_metadata.active value is not Boolean');
  }
  mdaas.disconnect();
});

test('update fails due to invalid _metadata.id', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: 1 });
  await mdaas.connect(configuration);
  await connectionPromise;
  try {
    mdaas.updateField('_metadata.id', 1);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe('_metadata.id value is not a string');
  }
  mdaas.disconnect();
});

test('update fails due to invalid path', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  await mdaas.connect(configuration);
  await connectionPromise;
  const path = 'a[0';
  try {
    mdaas.updateField(path, 1);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe(`"${path}" is not a valid field specifier`);
  }
  mdaas.disconnect();
});

test('update fails due to no field', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration();
  await mdaas.connect(configuration);
  await connectionPromise;
  const path = 'a.b';
  try {
    mdaas.updateField(path, 1);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe(`"${path}" does not specify a known field`);
  }
  mdaas.disconnect();
});

test('update fails due to out of bounds', async () => {
  expect.hasAssertions();
  const connectionPromise = createConsolePromise();
  const mdaas = createMDaaS();
  const configuration = createConfiguration({ a: [1] });
  await mdaas.connect(configuration);
  await connectionPromise;
  const path = 'a[1]';
  try {
    mdaas.updateField(path, 1);
    console.error('unexpected success');
  } catch(ex) {
    expect(ex.message).toBe(`"${path}" is out of bounds`);
  }
  mdaas.disconnect();
});

function createTokenExpiredPromise(configuration: Configuration, timeout?: number): Promise<void> {
  return new Promise((resolve, reject) => {
    const timerId = setTimeout(() => {
      clearHandlers();
      reject('timeout');
    }, timeout || 2222);
    configuration.onTokenExpired = (_fn: TokenRefreshFn) => {
      clearHandlers();
      resolve();
      return false;
    };

    function clearHandlers() {
      clearTimeout(timerId);
      configuration.onTokenExpired = (_fn: TokenRefreshFn) => {
        console.error('unexpected onTokenExpired invocation');
        return false;
      };
    }
  });
}

function createConsolePromise(timeout?: number): Promise<string> {
  return new Promise<string>((resolve, _reject) => {
    const error = console.error;
    const warn = console.warn;
    const timerId = setTimeout(() => {
      clearHandlers();
      resolve('timeout');
    }, timeout || 2222);
    console.error = (name: string, data: string) => {
      clearHandlers();
      resolve(name === '[MDaaS.onMessage]' ? data.startsWith('unexpected response from server:') ? data.split(' ').pop() : data : name);
    };
    console.warn = (data: string) => {
      clearHandlers();
      resolve(data);
    };

    function clearHandlers() {
      clearTimeout(timerId);
      console.error = error;
      console.warn = warn;
    }
  });
}

function createConfiguration(initialState?: State): Configuration {
  return {
    broadcasterIds: ['1'],
    environment: 'test',
    gameId: '1',
    initialState: initialState || {},
    token: '1',
    onTokenExpired: (_fn: TokenRefreshFn) => {
      console.error('unexpected token refresh call-back');
      return false;
    },
  };
}

function withMetadata(state: State): State {
  return { _metadata: { id: "test", active: true }, ...state };
}
